package com.proximities.sdk;


import android.Manifest;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Binder;
import android.os.IBinder;
import android.support.v4.content.ContextCompat;
import android.text.Html;
import android.widget.RemoteViews;

import com.bumptech.glide.Glide;
import com.bumptech.glide.request.animation.GlideAnimation;
import com.bumptech.glide.request.target.SimpleTarget;
import com.proximities.sdk.bridge.OnEventCallTransmitters;
import com.proximities.sdk.bridge.OnEventUserMoved;
import com.proximities.sdk.bridge.OnProximitiesService;
import com.proximities.sdk.database.AnimationsTable;
import com.proximities.sdk.database.DatabaseHandler;
import com.proximities.sdk.database.LogsTable;
import com.proximities.sdk.json.model.animation.AnimationData;
import com.proximities.sdk.json.model.log.AddedLog;
import com.proximities.sdk.json.model.log.AnimationLog;
import com.proximities.sdk.json.model.partner.BasePartner;
import com.proximities.sdk.json.model.partner.Partner;
import com.proximities.sdk.json.model.partner.Poi;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.BaseQrCode;
import com.proximities.sdk.json.model.transmitter.BaseTransmitter;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.message.EnableBluetooth;
import com.proximities.sdk.message.ProximitiesServiceReady;
import com.proximities.sdk.request.api.EntryExitLogRequest;
import com.proximities.sdk.request.api.LogRequest;
import com.proximities.sdk.request.api.TransmitterRequest;
import com.proximities.sdk.util.ApplicationUtil;
import com.proximities.sdk.util.LocationChange;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.proximities.sdk.util.Utils;

import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 24/12/14.
 *
 * This service is responsible for establishing and maintaining a
 * permanent connection between the device and user's position plus
 * beacon manager
 */
public class ProximitiesService extends Service implements BeaconConsumer, OnEventCallTransmitters,
        OnProximitiesService, OnEventUserMoved{

    private final IBinder mBinder = new MyBinder();

    private BeaconManager beaconManager;
    private boolean gpsManagerIsReady;
    private boolean beaconManagerIsReady;
    private boolean bluetoothIsReady;
    private static boolean allIsReady;
    private boolean areLogsSend = false;
    private ConnectionChangesReceiver connectionChangesReceiver = new ConnectionChangesReceiver();
    private int idCurrentNotification = 0;
    private boolean mIsIncompatibleDevice;

    @Override
    public void onCreate() {
        super.onCreate();
        mIsIncompatibleDevice = Utils.isIncompatibleDevice();
        registerReceiver(mReceiver, new IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED));
        registerReceiver(connectionChangesReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
        beaconManager = BeaconManager.getInstanceForApplication(this);
        EventBus.getDefault().register(this);
    }

    public void setGpsManagerIsReady(boolean gpsManagerIsReady) {
        this.gpsManagerIsReady = gpsManagerIsReady;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        if (intent != null) {  // First launch [launched by app]
            // Toast.makeText(this, "Service is running", Toast.LENGTH_SHORT).show();

            if (ProximitiesGpsManager.getInstance().isConnected()) {
                gpsManagerIsReady = true;
            }

            if (beaconManager.isBound(this)) {
                beaconManagerIsReady = true;
            }

            if (!bluetoothIsEnabled() && intent.getBooleanExtra(ProximitiesConstants.EXTRA_CHECK_BLE_IS_ENABLED, false)) {
                if (ProximitiesPrefs.readAskBleAgain(this)) {
                    Intent i = new Intent(this, BluetoothActivity.class);
                    i.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    startActivity(i);
                } else {
                    bluetoothIsReady = true;
                }
            } else {
                bluetoothIsReady = true;
            }
            checkAllIsReady();
        } else {  // Recreated by system
            beaconManager.setBackgroundMode(true);
        }
        if(ContextCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
            initGpsManager();
            initBeaconManager();
        }
        // We want this service to continue running until it is explicitly
        // stopped, so return sticky.
        return START_STICKY;
    }

    private void initBeaconManager(){
        if(!mIsIncompatibleDevice) ProximitiesBeaconManager.getInstance().initBeaconManager(getApplicationContext(), beaconManager, this, this);
    }

    private void initGpsManager(){
        ProximitiesGpsManager.getInstance().initLocationManager(getApplication(), this, this);
    }

    /**
     * Check if bluetooth is available and activated
     *
     * @return true if bluetooth is enabled, false otherwise
     */
    private boolean bluetoothIsEnabled() {
        BluetoothAdapter mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        if (mBluetoothAdapter != null && mBluetoothAdapter.isEnabled()) {
            return true;
        }
        return false;
    }

    @Override
    public void onDestroy() {
        //ProximitiesGpsManager.getInstance().stop();
        beaconManager.unbind(this);
        unregisterReceiver(mReceiver);
        unregisterReceiver(connectionChangesReceiver);
        EventBus.getDefault().unregister(this);
        ProximitiesGpsManager.getInstance().removeLocationUpdates();
        // Tell the user we stopped.
        // Toast.makeText(this, "Service is stopped", Toast.LENGTH_SHORT).show();
        super.onDestroy();
    }

    @Override
    public IBinder onBind(Intent intent) {
        return mBinder;
    }

    public void startAnim(ArrayList<Campaign> anims) {
        Intent intent = new Intent(this, ProximitiesAnimationActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        intent.putExtra(ProximitiesConstants.EXTRA_ANIMS, anims);

        Campaign anim = anims.get(0);
        if (anims.size() == 1 && anim.getTemplateType() != null && anim.getTemplateType().equals(ProximitiesConstants.SIMPLE_NOTIF)) {
            if(ApplicationUtil.getMyInstance().isResumed()) {
                startActivity(intent);
            } else {
                intent.putExtra(ProximitiesConstants.IS_BACKGROUND, true);
                generatePushOrDelegate(intent, anim);
                manageLog(anims, ProximitiesConstants.ANIM_RECEIVED);
            }
        } else if (ApplicationUtil.getMyInstance().isResumed()) {
            startActivity(intent);
        } else {
            if (anims.size() > 1) {
                anim = null;  // multiple anim
            }
            generatePushOrDelegate(intent, anim);
            manageLog(anims, ProximitiesConstants.ANIM_RECEIVED);
        }
    }

    /**
     * Generate a push or let user to generate push
     *
     * @param intent {@link android.content.Intent} to run when user clicks on notif
     * @param anim {@link Campaign} or null if there are multiple available animation
     */
    private void generatePushOrDelegate(final Intent intent, final Campaign anim) {
        intent.putExtra(ProximitiesConstants.ANIM_BY_PUSH, true);

        if (anim != null){

            final RemoteViews contentView = new RemoteViews(getPackageName(), R.layout.notification_layout);

            if(anim.getPoi() != null && !anim.getPoi().getName().isEmpty())
                contentView.setTextViewText(R.id.poi_name, anim.getPoi().getName());
            if(anim.getTitle() != null && !anim.getTitle().isEmpty())
                contentView.setTextViewText(R.id.content, Html.fromHtml(anim.getTitle()).toString());

            contentView.setImageViewResource(R.id.logo, ProximitiesConfig.getNotificationIcon(this));
            if(anim.getImage() != null && !anim.getImage().isEmpty()) {
                Glide.with(this).load(ProximitiesConstants.STATIC_CONTENT_HOST + anim.getImage())
                        .asBitmap()
                        .into(new SimpleTarget<Bitmap>() {
                            @Override
                            public void onResourceReady(Bitmap resource, GlideAnimation glideAnimation) {
                                contentView.setImageViewBitmap(R.id.image, resource);
                                createNotification(Html.fromHtml(anim.getTitle()).toString(), intent, contentView, resource);
                            }

                            @Override
                            public void onLoadFailed(Exception e, Drawable errorDrawable) {
                                createNotification(Html.fromHtml(anim.getTitle()).toString(), intent, null, null);
                                super.onLoadFailed(e, errorDrawable);
                            }
                        });
            } else {
                createNotification(Html.fromHtml(anim.getTitle()).toString(), intent, null, null);
            }
        } else {
            createNotification(Html.fromHtml(getString(ProximitiesConfig.getNotificationContent(this))).toString(), intent, null, null);
        }
    }

    private void createNotification(String msg, Intent intent, RemoteViews contentView, Bitmap bitmap) {
        Notification notif = new Notification.Builder(this)
                .setSmallIcon(ProximitiesConfig.getNotificationIcon(this))
                .setLargeIcon((bitmap == null) ? (BitmapFactory.decodeResource(this.getResources(), ProximitiesConfig.getNotificationIcon(this))) : bitmap)
                .setPriority(Notification.PRIORITY_HIGH)
                .setContentTitle(getString(R.string.app_name))
                .setContentText(msg)
                .build();

        if(contentView != null)
            notif.bigContentView = contentView;
        notif.contentIntent = PendingIntent.getActivity(this, idCurrentNotification, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        notif.vibrate = new long[]{1000, 1000};
        notif.ledARGB = ContextCompat.getColor(this, ProximitiesConfig.getNotificationLightColor(this));
        notif.flags = Notification.FLAG_SHOW_LIGHTS;
        notif.flags = Notification.FLAG_AUTO_CANCEL;
        notif.ledOnMS = 1000;
        notif.ledOffMS = 2500;

        NotificationManager mNotifyManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
        mNotifyManager.notify(idCurrentNotification, notif);
        idCurrentNotification++;
    }

    private void manageLog(List<Campaign> campaigns, String userAction) {
        AnimationLog animLog = new AnimationLog();
        animLog.setUserAction(userAction);
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        animLog.setCreated(format.format(calendar.getTime()));
        List<Integer> animIds = new ArrayList<>();
        for (Campaign anim : campaigns) {
            animIds.add(anim.getId());
            animLog.setAnimationsIds(animIds);
        }
        LogRequest logReq = LogRequest.getInstance(getApplication());
        logReq.executePost(animLog);
    }

    @Override
    public void onCallTransmitters(String uuid, String major, String minor, String namespace, String instance) {
        TransmitterRequest transmitterReq = TransmitterRequest.getInstance(getApplication(), this);
        if(namespace != null){
            transmitterReq.executeGet(namespace, instance);
        } else {
            transmitterReq.executeGet(uuid, major, minor);
        }
    }

    @Override
    public void onEntryLogTransmitter(String beaconLog) {
        String[] beaconDetails = splitBeaconLog(beaconLog);
        if(beaconDetails.length == 3) {
            EntryExitLogRequest entryReq = EntryExitLogRequest.getInstance(getApplication());
            entryReq.executeEntryPost(beaconDetails);
        }
    }

    @Override
    public void onExitLogTransmitter(String beaconLog) {
        String[] beaconDetails = splitBeaconLog(beaconLog);
        if(beaconDetails.length == 3) {
            EntryExitLogRequest exitReq = EntryExitLogRequest.getInstance(getApplication());
            exitReq.executeExitPost(beaconDetails);
        }
    }

    private String[] splitBeaconLog(String beaconLog){
        return beaconLog.split("\\-\\-");
    }

    @Override
    public void onGpsReadyWithRealCoord(boolean areDefaultCoordinates) {
        gpsManagerIsReady = true;
        checkAllIsReady();
    }

    private void checkAllIsReady() {
        if (gpsManagerIsReady && beaconManagerIsReady && bluetoothIsReady && !allIsReady) {
            EventBus.getDefault().post(ProximitiesServiceReady.getInstance());
            allIsReady = true;  // disable check above
        }
    }

    @Override
    public void onUserMoved(boolean withGPS) {
    }

    public class MyBinder extends Binder {
        public ProximitiesService getService() {
            return ProximitiesService.this;
        }
    }

    @Override
    public void onBeaconServiceConnect() {
        if(!mIsIncompatibleDevice) {
            ProximitiesBeaconManager.getInstance().startMonitoring();
            ProximitiesBeaconManager.getInstance().startRanging();
            beaconManagerIsReady = true;
            checkAllIsReady();
        }
    }

    /**
     * Callback on GET {@link com.proximities.sdk.util.ProximitiesConstants#WS_TRANSMITTERS} with success
     *
     * @param data  {@link BaseTransmitter} object
     * @see #onCallTransmitters(String, String, String, String, String)
     * @see TransmitterRequest
     */
    @Subscribe
    public void onEventMainThread(BaseTransmitter data) {
        try {
            if (!ProximitiesPrefs.readDisableAnim(this)) {
                if (data.getData().getTransmitters().size() > 0) {
                    ArrayList<Campaign> anims = (ArrayList<Campaign>) data.getData().getTransmitters().get(0).getCampaigns();
                    anims.get(0).setPoi(data.getData().getTransmitters().get(0).getPoi().get(0));
                    if (anims.size() > 0 && !data.getData().getTransmitters().get(0).getType().equalsIgnoreCase("nfc")) {
                        DatabaseHandler dbHandler = new DatabaseHandler(this);
                        SQLiteDatabase db = dbHandler.getReadableDatabase();
                        if (!dbHandler.isAnimationInLogs(data.getData().getTransmitters().get(0).getCampaigns().get(0).getId(), db)) {
                            /*if(data.getData().getTransmitters().get(0).getPoi().get(0).getBackgroundFetch() == 1){
                                dbHandler.insertLogsIntoDatabase(data.getData().getTransmitters().get(0).getCampaigns().get(0).getId(), ProximitiesConstants.ANIM_RECEIVED);
                            }*/
                            this.startAnim(anims);
                        }
                        db.close();
                        dbHandler.close();
                    }
                }
            }
        } catch (Exception e){
            e.printStackTrace();
        }
    }

    @Subscribe
    public void onEventMainThread(BaseQrCode data) {
        try {
            if (!ProximitiesPrefs.readDisableAnim(this)) {
                if (data.getData().getTransmitters().size() > 0) {
                    ArrayList<Campaign> anims = (ArrayList<Campaign>) data.getData().getTransmitters().get(0).getCampaigns();
                    anims.get(0).setPoi(data.getData().getTransmitters().get(0).getPoi().get(0));
                    this.startAnim(anims);
                } else {
                    if(ProximitiesConfig.getOnResponseFromQrScanListener() != null) ProximitiesConfig.getOnResponseFromQrScanListener().onNoCampaignResponse();
                }
            }
        } catch (Exception e){
            e.printStackTrace();
        }
    }

    /**
     * Callback on GET {@link com.proximities.sdk.util.ProximitiesConstants#WS_ANIM} with success
     *
     * @param data  {@link com.proximities.sdk.json.model.animation.AnimationData} object
     */
    @Subscribe
    public void onEventMainThread(AnimationData data) {
        ArrayList<Campaign> campaigns = new ArrayList<>();
        if (data != null && data.getData() != null) campaigns.addAll(data.getData().getCampaigns());
        ProximitiesConfig.setFavorites(campaigns);
    }

    @Subscribe
    public void onEventMainThread(EnableBluetooth data) {
        bluetoothIsReady = true;
        checkAllIsReady();
    }

    /*******************************************************************************************************/
    /*********************************** GEOFENCING ********************************************************/
    /*******************************************************************************************************/


    private Map<String, ArrayList<Map<String, String>>> manageLogInDatabase(SQLiteDatabase db){
        if(!areLogsSend) {
            areLogsSend = true;
            String query = "SELECT * FROM " + LogsTable.LogsEntry.TABLE_NAME + ";";
            Cursor cursor = db.rawQuery(query, null);
            if (cursor.moveToFirst()) {
                Map<String, ArrayList<Map<String, String>>> body = new HashMap<>();
                ArrayList<Map<String, String>> logs = new ArrayList<>();
                do {
                    Map<String, String> log = new HashMap<>();
                    log.put("id", cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_ANIM_ID)));
                    log.put("created", cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_CREATED)));
                    log.put("action", cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_ACTION)));
                    logs.add(log);
                } while (cursor.moveToNext());
                body.put("animationsLogs", logs);
                cursor.close();
                return body;
            } else {
                cursor.close();
                return null;
            }
        } else {
            return null;
        }
    }

    /**
     * Callback on POST {@link com.proximities.sdk.util.ProximitiesConstants#WS_LOGS} with success
     *
     * @param data  {@link AddedLog} object
     * @see com.proximities.sdk.request.api.LogRequest
     */
    @Subscribe
    public void onEventMainThread(AddedLog data) {
        DatabaseHandler dbHandler = new DatabaseHandler(this);
        SQLiteDatabase db = dbHandler.getWritableDatabase();
        db.delete(LogsTable.LogsEntry.TABLE_NAME, null, null);
        db.close();
        dbHandler.close();
        areLogsSend = false;
    }

    /**
     * Callback on GET {@link com.proximities.sdk.util.ProximitiesConstants#WS_PARTNER} with success
     *
     * @param data  {@link com.proximities.sdk.json.model.partner.BasePartner} object
     * @see com.proximities.sdk.request.api.PartnerRequest
     */
    @Subscribe
    public void onEventMainThread(BasePartner data) {
        if (!ProximitiesPrefs.readDisableAnim(this)) {
            if (data.isCalledFromGpsOrBeacon()) {
                DatabaseHandler dbHandler = new DatabaseHandler(this);
                SQLiteDatabase  db = dbHandler.getReadableDatabase();
                LogRequest logReq = LogRequest.getInstance(getApplication());
                logReq.executePostFromDatabase(manageLogInDatabase(db));
                db = dbHandler.getWritableDatabase();
                db.delete(AnimationsTable.AnimationsEntry.TABLE_NAME, null, null);
                List<Campaign> myCampaigns = new ArrayList<>();
                List<Partner> partners = data.getData().getPartners();
                if (partners != null && !partners.isEmpty()) {
                    for (Partner p : partners) {
                        List<Poi> pois = p.getPois();
                        if (pois != null && !pois.isEmpty()) {
                            for (Poi poi : pois) {
                                List<Transmitter> transmitters = poi.getTransmitters();
                                if (transmitters != null && !transmitters.isEmpty()) {
                                    for (Transmitter trans : transmitters) {
                                        if(trans.getCampaigns() != null){
                                            if(poi.getBackgroundFetch() == 1) {
                                                db = dbHandler.getReadableDatabase();
                                                if (!dbHandler.isAnimationInLogs(trans.getCampaigns().get(0).getId(), db)) {
                                                    db = dbHandler.getWritableDatabase();
                                                    dbHandler.insertAnimationsInDatabase(trans, db);
                                                }
                                            }
                                        }
                                        if (trans.getType().equals("beacon")) {
                                            if(!mIsIncompatibleDevice) ProximitiesBeaconManager.getInstance().addRegion(trans.getUuid());
                                        }
                                    }
                                }
                                if (poi.getCampaigns() != null) {
                                    List<Campaign> campaigns = poi.getCampaigns();
                                    for (Campaign a : campaigns){
                                        a.setPoi(poi);
                                    }
                                    myCampaigns.addAll(campaigns);
                                }

                                if(pois.size() == 1){
                                    ProximitiesGpsManager.getInstance().updateLocationRequest(pois.get(0).getDistance());
                                }
                            }
                        }
                    }
                }
                if(!mIsIncompatibleDevice) {
                    ProximitiesBeaconManager.getInstance().startMonitoring();
                    ProximitiesBeaconManager.getInstance().startRanging();
                }
                if (myCampaigns.size() > 0) {
                    this.startAnim((ArrayList) myCampaigns);
                }
                db.close();
                dbHandler.close();
            }
        }
    }

    @Subscribe
    public void onEventMainThread(LocationChange data) {
        if(ProximitiesConfig.getOnLocationChangeListener() != null) ProximitiesConfig.getOnLocationChangeListener().onLocationChange();
    }

    /*******************************************************************************************************/
    /*********************************** RECEIVERS ********************************************************/
    /*******************************************************************************************************/


    private final BroadcastReceiver mReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            final String action = intent.getAction();

            if (action.equals(BluetoothAdapter.ACTION_STATE_CHANGED)) {
                final int state = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE,
                        BluetoothAdapter.ERROR);
                switch (state) {
                    case BluetoothAdapter.STATE_ON:
                        initBeaconManager();
                        break;
                }
            }
        }
    };

    private class ConnectionChangesReceiver extends BroadcastReceiver {

        private final String CONNECTIVITY_CHANGES_ACTION = "android.net.conn.CONNECTIVITY_CHANGE";

        public ConnectionChangesReceiver(){

        }

        @Override
        public void onReceive(Context context, Intent intent) {
            if (intent.getAction().equalsIgnoreCase(CONNECTIVITY_CHANGES_ACTION)){
                ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
                boolean isConnected = activeNetwork != null && activeNetwork.isConnectedOrConnecting();
                if(isConnected) {
                    initGpsManager();
                }
            }
        }
    };

}
