package org.stellar.sdk;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.BaseEncoding;
import org.stellar.sdk.xdr.DecoratedSignature;
import org.stellar.sdk.xdr.Signature;
import org.stellar.sdk.xdr.SignatureHint;

import java.io.IOException;
import java.security.SecureRandom;
import java.util.*;

public class Sep10Challenge {
  private static final String MANAGER_DATA_NAME_FLAG = "auth";
  /**
   * Returns a valid <a href="https://github.com/stellar/stellar-protocol/blob/master/ecosystem/sep-0010.md#response" target="_blank">SEP 10</a> challenge, for use in web authentication.
   * @param signer           The server's signing account.
   * @param network          The Stellar network used by the server.
   * @param clientAccountId  The stellar account belonging to the client.
   * @param domainName       The <a href="https://en.wikipedia.org/wiki/Fully_qualified_domain_name" target="_blank">fully qualified domain name</a> of the service requiring authentication.
   * @param timebounds       The lifetime of the challenge token.
   */
  public static Transaction newChallenge(
      KeyPair signer,
      Network network,
      String clientAccountId,
      String domainName,
      TimeBounds timebounds
  ) throws InvalidSep10ChallengeException {
    byte[] nonce = new byte[48];
    SecureRandom random = new SecureRandom();
    random.nextBytes(nonce);
    BaseEncoding base64Encoding = BaseEncoding.base64();
    byte[] encodedNonce = base64Encoding.encode(nonce).getBytes();


    if (StrKey.decodeVersionByte(clientAccountId) != StrKey.VersionByte.ACCOUNT_ID) {
      throw new InvalidSep10ChallengeException(clientAccountId+" is not a valid account id");
    }

    Account sourceAccount = new Account(signer.getAccountId(), -1L);
    ManageDataOperation operation = new ManageDataOperation.Builder(String.format("%s %s", domainName, MANAGER_DATA_NAME_FLAG), encodedNonce)
        .setSourceAccount(clientAccountId)
        .build();
    Transaction transaction = new Transaction.Builder(sourceAccount, network)
        .addTimeBounds(timebounds)
        .setBaseFee(100)
        .addOperation(operation)
        .build();

    transaction.sign(signer);

    return transaction;
  }

  /**
   * Reads a SEP 10 challenge transaction and returns the decoded transaction envelope and client account ID contained within.
   * <p>
   * It also verifies that transaction is signed by the server.
   * <p>
   * It does not verify that the transaction has been signed by the client or
   * that any signatures other than the servers on the transaction are valid. Use
   * one of the following functions to completely verify the transaction:
   * {@link Sep10Challenge#verifyChallengeTransactionSigners(String, String, Network, String, Set)} or
   * {@link Sep10Challenge#verifyChallengeTransactionThreshold(String, String, Network, String, int, Set)} or
   * {@link Sep10Challenge#verifyChallengeTransactionSigners(String, String, Network, String[], Set)} or
   * {@link Sep10Challenge#verifyChallengeTransactionThreshold(String, String, Network, String[], int, Set)} or
   *
   * @param challengeXdr    SEP-0010 transaction challenge transaction in base64.
   * @param serverAccountId Account ID for server's account.
   * @param network         The network to connect to for verifying and retrieving.
   * @param domainNames     An array of home domains, one of which is expected to be included in the first Manage Data operation's string key.
   * @return {@link ChallengeTransaction}, the decoded transaction envelope and client account ID contained within.
   * @throws InvalidSep10ChallengeException If the SEP-0010 validation fails, the exception will be thrown.
   * @throws IOException                    If read XDR string fails, the exception will be thrown.
   */
  public static ChallengeTransaction readChallengeTransaction(String challengeXdr, String serverAccountId, Network network, String[] domainNames) throws InvalidSep10ChallengeException, IOException {
    if (domainNames == null || domainNames.length == 0) {
      throw new IllegalArgumentException("At least one domain name must be included in domainNames.");
    }

    // decode the received input as a base64-urlencoded XDR representation of Stellar transaction envelope
    AbstractTransaction parsed = Transaction.fromEnvelopeXdr(challengeXdr, network);
    if (!(parsed instanceof Transaction)) {
      throw new InvalidSep10ChallengeException("Transaction cannot be a fee bump transaction");
    }
    Transaction transaction = (Transaction)parsed;

    if (StrKey.decodeVersionByte(serverAccountId) != StrKey.VersionByte.ACCOUNT_ID) {
      throw new InvalidSep10ChallengeException("serverAccountId: "+serverAccountId+" is not a valid account id");
    }

    // verify that transaction source account is equal to the server's signing key
    if (!serverAccountId.equals(transaction.getSourceAccount())) {
      throw new InvalidSep10ChallengeException("Transaction source account is not equal to server's account.");
    }

    // verify that transaction sequenceNumber is equal to zero
    if (transaction.getSequenceNumber() != 0L) {
      throw new InvalidSep10ChallengeException("The transaction sequence number should be zero.");
    }

    // verify that transaction has time bounds set, and that current time is between the minimum and maximum bounds.
    if (transaction.getTimeBounds() == null) {
      throw new InvalidSep10ChallengeException("Transaction requires timebounds.");
    }

    long maxTime = transaction.getTimeBounds().getMaxTime();
    long minTime = transaction.getTimeBounds().getMinTime();
    if (maxTime == 0L) {
      throw new InvalidSep10ChallengeException("Transaction requires non-infinite timebounds.");
    }

    long currentTime = System.currentTimeMillis() / 1000L;
    if (currentTime < minTime || currentTime > maxTime) {
      throw new InvalidSep10ChallengeException("Transaction is not within range of the specified timebounds.");
    }

    if (transaction.getOperations().length < 1) {
      throw new InvalidSep10ChallengeException("Transaction requires at least one ManageData operation.");
    }

    // verify that the first operation in the transaction is a Manage Data operation
    // and its source account is not null
    Operation operation = transaction.getOperations()[0];
    if (!(operation instanceof ManageDataOperation)) {
      throw new InvalidSep10ChallengeException("Operation type should be ManageData.");
    }
    ManageDataOperation manageDataOperation = (ManageDataOperation) operation;

    // verify that transaction envelope has a correct signature by server's signing key
    String clientAccountId = manageDataOperation.getSourceAccount();
    if (clientAccountId == null) {
      throw new InvalidSep10ChallengeException("Operation should have a source account.");
    }

    String matchedDomainName = null;
    for (String homeDomain : domainNames) {
      if ((homeDomain + " " + MANAGER_DATA_NAME_FLAG).equals(manageDataOperation.getName())) {
        matchedDomainName = homeDomain;
        break;
      }
    }

    if (matchedDomainName == null) {
      throw new InvalidSep10ChallengeException("The transaction's operation key name does not include one of the expected home domains.");
    }

    if (StrKey.decodeVersionByte(clientAccountId) != StrKey.VersionByte.ACCOUNT_ID) {
      throw new InvalidSep10ChallengeException("clientAccountId: "+clientAccountId+" is not a valid account id");
    }

    if (manageDataOperation.getValue() == null) {
      throw new InvalidSep10ChallengeException("The transaction's operation value should not be null.");
    }
    // verify manage data value
    if (manageDataOperation.getValue().length != 64) {
      throw new InvalidSep10ChallengeException("Random nonce encoded as base64 should be 64 bytes long.");
    }

    BaseEncoding base64Encoding = BaseEncoding.base64();
    byte[] nonce;
    try {
      nonce = base64Encoding.decode(new String(manageDataOperation.getValue()));
    } catch (IllegalArgumentException e) {
      throw new InvalidSep10ChallengeException("Failed to decode random nonce provided in ManageData operation.", e);
    }

    if (nonce.length != 48) {
      throw new InvalidSep10ChallengeException("Random nonce before encoding as base64 should be 48 bytes long.");
    }

    // verify subsequent operations are manage data ops with source account set to server account
    for (int i = 1; i < transaction.getOperations().length; i++) {
      Operation op = transaction.getOperations()[i];
      if (!(op instanceof ManageDataOperation)) {
        throw new InvalidSep10ChallengeException("Operation type should be ManageData.");
      }
      ManageDataOperation manageDataOp = (ManageDataOperation) op;
      if (manageDataOp.getSourceAccount() == null) {
        throw new InvalidSep10ChallengeException("Operation should have a source account.");
      }
      if (!manageDataOp.getSourceAccount().equals(serverAccountId)) {
        throw new InvalidSep10ChallengeException("Subsequent operations are unrecognized.");
      }
    }

    if (!verifyTransactionSignature(transaction, serverAccountId)) {
      throw new InvalidSep10ChallengeException(String.format("Transaction not signed by server: %s.", serverAccountId));
    }

    return new ChallengeTransaction(transaction, clientAccountId, matchedDomainName);
  }

  /**
   * Reads a SEP 10 challenge transaction and returns the decoded transaction envelope and client account ID contained within.
   * <p>
   * It also verifies that transaction is signed by the server.
   * <p>
   * It does not verify that the transaction has been signed by the client or
   * that any signatures other than the servers on the transaction are valid. Use
   * one of the following functions to completely verify the transaction:
   * {@link Sep10Challenge#verifyChallengeTransactionSigners(String, String, Network, String, Set)} or
   * {@link Sep10Challenge#verifyChallengeTransactionThreshold(String, String, Network, String, int, Set)} or
   * {@link Sep10Challenge#verifyChallengeTransactionSigners(String, String, Network, String[], Set)} or
   * {@link Sep10Challenge#verifyChallengeTransactionThreshold(String, String, Network, String[], int, Set)} or
   *
   * @param challengeXdr    SEP-0010 transaction challenge transaction in base64.
   * @param serverAccountId Account ID for server's account.
   * @param network         The network to connect to for verifying and retrieving.
   * @param domainName      The home domain that is expected to be included in the first Manage Data operation's string key.
   * @return {@link ChallengeTransaction}, the decoded transaction envelope and client account ID contained within.
   * @throws InvalidSep10ChallengeException If the SEP-0010 validation fails, the exception will be thrown.
   * @throws IOException                    If read XDR string fails, the exception will be thrown.
   */
  public static ChallengeTransaction readChallengeTransaction(String challengeXdr, String serverAccountId, Network network, String domainName) throws InvalidSep10ChallengeException, IOException {
    return readChallengeTransaction(challengeXdr, serverAccountId, network, new String[]{domainName});
  }

  /**
   * Verifies that for a SEP 10 challenge transaction
   * all signatures on the transaction are accounted for. A transaction is
   * verified if it is signed by the server account, and all other signatures
   * match a signer that has been provided as an argument. Additional signers can
   * be provided that do not have a signature, but all signatures must be matched
   * to a signer for verification to succeed. If verification succeeds a list of
   * signers that were found is returned, excluding the server account ID.
   *
   * @param challengeXdr    SEP-0010 transaction challenge transaction in base64.
   * @param serverAccountId Account ID for server's account.
   * @param network         The network to connect to for verifying and retrieving.
   * @param domainName      The home domain that is expected to be included in the first Manage Data operation's string key.
   * @param signers         The signers of client account.
   * @return a list of signers that were found is returned, excluding the server account ID.
   * @throws InvalidSep10ChallengeException If the SEP-0010 validation fails, the exception will be thrown.
   * @throws IOException                    If read XDR string fails, the exception will be thrown.
   */
  public static Set<String> verifyChallengeTransactionSigners(String challengeXdr, String serverAccountId, Network network, String domainName, Set<String> signers) throws InvalidSep10ChallengeException, IOException {
    return verifyChallengeTransactionSigners(challengeXdr, serverAccountId, network, new String[]{domainName}, signers);
  }

  /**
   * Verifies that for a SEP 10 challenge transaction
   * all signatures on the transaction are accounted for. A transaction is
   * verified if it is signed by the server account, and all other signatures
   * match a signer that has been provided as an argument. Additional signers can
   * be provided that do not have a signature, but all signatures must be matched
   * to a signer for verification to succeed. If verification succeeds a list of
   * signers that were found is returned, excluding the server account ID.
   *
   * @param challengeXdr    SEP-0010 transaction challenge transaction in base64.
   * @param serverAccountId Account ID for server's account.
   * @param network         The network to connect to for verifying and retrieving.
   * @param domainNames     An array of home domains, one of which is expected to be included in the first Manage Data operation's string key.
   * @param signers         The signers of client account.
   * @return a list of signers that were found is returned, excluding the server account ID.
   * @throws InvalidSep10ChallengeException If the SEP-0010 validation fails, the exception will be thrown.
   * @throws IOException                    If read XDR string fails, the exception will be thrown.
   */
  public static Set<String> verifyChallengeTransactionSigners(String challengeXdr, String serverAccountId, Network network, String[] domainNames, Set<String> signers) throws InvalidSep10ChallengeException, IOException {
    if (signers == null || signers.isEmpty()) {
      throw new InvalidSep10ChallengeException("No verifiable signers provided, at least one G... address must be provided.");
    }

    // Read the transaction which validates its structure.
    ChallengeTransaction parsedChallengeTransaction = readChallengeTransaction(challengeXdr, serverAccountId, network, domainNames);
    Transaction transaction = parsedChallengeTransaction.getTransaction();

    // Ensure the server account ID is an address and not a seed.
    KeyPair serverKeyPair = KeyPair.fromAccountId(serverAccountId);

    // Deduplicate the client signers and ensure the server is not included
    // anywhere we check or output the list of signers.
    Set<String> clientSigners = new HashSet<String>();
    for (String signer : signers) {
      // Ignore non-G... account/address signers.
      StrKey.VersionByte versionByte;
      try {
        versionByte = StrKey.decodeVersionByte(signer);
      } catch (Exception e) {
        continue;
      }

      if (!StrKey.VersionByte.ACCOUNT_ID.equals(versionByte)) {
        continue;
      }

      // Ignore the server signer if it is in the signers list. It's
      // important when verifying signers of a challenge transaction that we
      // only verify and return client signers. If an account has the server
      // as a signer the server should not play a part in the authentication
      // of the client.
      if (serverKeyPair.getAccountId().equals(signer)) {
        continue;
      }
      clientSigners.add(signer);
    }

    // Don't continue if none of the signers provided are in the final list.
    if (clientSigners.isEmpty()) {
      throw new InvalidSep10ChallengeException("No verifiable signers provided, at least one G... address must be provided.");
    }

    // Verify all the transaction's signers (server and client) in one
    // hit. We do this in one hit here even though the server signature was
    // checked in the readChallengeTx to ensure that every signature and signer
    // are consumed only once on the transaction.
    Set<String> allSigners = new HashSet<String>(clientSigners);
    allSigners.add(serverKeyPair.getAccountId());
    Set<String> signersFound = verifyTransactionSignatures(transaction, allSigners);

    // Confirm the server is in the list of signers found and remove it.
    boolean serverSignerFound = signersFound.remove(serverKeyPair.getAccountId());

    // Confirm we matched a signature to the server signer.
    if (!serverSignerFound) {
      throw new InvalidSep10ChallengeException(String.format("Transaction not signed by server: %s.", serverAccountId));
    }

    // Confirm we matched signatures to the client signers.
    if (signersFound.isEmpty()) {
      throw new InvalidSep10ChallengeException("Transaction not signed by any client signer.");
    }

    // Confirm all signatures were consumed by a signer.
    if (signersFound.size() != transaction.getSignatures().size() - 1) {
      throw new InvalidSep10ChallengeException("Transaction has unrecognized signatures.");
    }

    return signersFound;
  }

  /**
   * Verifies that for a SEP-0010 challenge transaction
   * all signatures on the transaction are accounted for and that the signatures
   * meet a threshold on an account. A transaction is verified if it is signed by
   * the server account, and all other signatures match a signer that has been
   * provided as an argument, and those signatures meet a threshold on the
   * account.
   *
   * @param challengeXdr    SEP-0010 transaction challenge transaction in base64.
   * @param serverAccountId Account ID for server's account.
   * @param network         The network to connect to for verifying and retrieving.
   * @param domainNames     An array of home domains, one of which is expected to be included in the first Manage Data operation's string key.
   * @param threshold       The threshold on the client account.
   * @param signers         The signers of client account.
   * @return a list of signers that were found is returned, excluding the server account ID.
   * @throws InvalidSep10ChallengeException If the SEP-0010 validation fails, the exception will be thrown.
   * @throws IOException                    If read XDR string fails, the exception will be thrown.
   */
  public static Set<String> verifyChallengeTransactionThreshold(String challengeXdr, String serverAccountId, Network network, String[] domainNames, int threshold, Set<Signer> signers) throws InvalidSep10ChallengeException, IOException {
    if (signers == null || signers.isEmpty()) {
      throw new InvalidSep10ChallengeException("No verifiable signers provided, at least one G... address must be provided.");
    }

    Map<String, Integer> weightsForSigner = new HashMap<String, Integer>();
    for (Signer signer : signers) {
      weightsForSigner.put(signer.getKey(), signer.getWeight());
    }

    Set<String> signersFound = verifyChallengeTransactionSigners(challengeXdr, serverAccountId, network, domainNames, weightsForSigner.keySet());

    int sum = 0;
    for (String signer : signersFound) {
      Integer weight = weightsForSigner.get(signer);
      if (weight != null) {
        sum += weight;
      }
    }

    if (sum < threshold) {
      throw new InvalidSep10ChallengeException(String.format("Signers with weight %d do not meet threshold %d.", sum, threshold));
    }

    return signersFound;
  }

  /**
   * Verifies that for a SEP-0010 challenge transaction
   * all signatures on the transaction are accounted for and that the signatures
   * meet a threshold on an account. A transaction is verified if it is signed by
   * the server account, and all other signatures match a signer that has been
   * provided as an argument, and those signatures meet a threshold on the
   * account.
   *
   * @param challengeXdr    SEP-0010 transaction challenge transaction in base64.
   * @param serverAccountId Account ID for server's account.
   * @param network         The network to connect to for verifying and retrieving.
   * @param domainName      The home domain that is expected to be included in the first Manage Data operation's string key.
   * @param threshold       The threshold on the client account.
   * @param signers         The signers of client account.
   * @return a list of signers that were found is returned, excluding the server account ID.
   * @throws InvalidSep10ChallengeException If the SEP-0010 validation fails, the exception will be thrown.
   * @throws IOException                    If read XDR string fails, the exception will be thrown.
   */
  public static Set<String> verifyChallengeTransactionThreshold(String challengeXdr, String serverAccountId, Network network, String domainName, int threshold, Set<Signer> signers) throws InvalidSep10ChallengeException, IOException {
    return verifyChallengeTransactionThreshold(challengeXdr, serverAccountId, network, new String[]{domainName}, threshold, signers);
  }

  private static Set<String> verifyTransactionSignatures(Transaction transaction, Set<String> signers) throws InvalidSep10ChallengeException {
    if (transaction.getSignatures().isEmpty()) {
      throw new InvalidSep10ChallengeException("Transaction has no signatures.");
    }

    byte[] txHash = transaction.hash();

    // find and verify signatures
    Set<String> signersFound = new HashSet<String>();
    Multimap<SignatureHint, Signature> signatures = HashMultimap.create();
    for (DecoratedSignature decoratedSignature : transaction.getSignatures()) {
      signatures.put(decoratedSignature.getHint(), decoratedSignature.getSignature());
    }

    for (String signer : signers) {
      KeyPair keyPair = KeyPair.fromAccountId(signer);
      SignatureHint hint = keyPair.getSignatureHint();

      for (Signature signature : signatures.get(hint)) {
        if (keyPair.verify(txHash, signature.getSignature())) {
          signersFound.add(signer);
          // explicitly ensure that a transaction signature cannot be
          // mapped to more than one signer
          signatures.remove(hint, signature);
          break;
        }
      }
    }

    return signersFound;
  }

  private static boolean verifyTransactionSignature(Transaction transaction, String accountId) throws InvalidSep10ChallengeException {
    return !verifyTransactionSignatures(transaction, Collections.singleton(accountId)).isEmpty();
  }

  /**
   * Used to store the results produced by {@link Sep10Challenge#readChallengeTransaction(String, String, Network, String[])}.
   */
  public static class ChallengeTransaction {
    private final Transaction transaction;
    private final String clientAccountId;
    private final String matchedHomeDomain;

    public ChallengeTransaction(Transaction transaction, String clientAccountId, String matchedHomeDomain) {
      this.transaction = transaction;
      this.clientAccountId = clientAccountId;
      this.matchedHomeDomain = matchedHomeDomain;
    }

    public Transaction getTransaction() {
      return transaction;
    }

    public String getClientAccountId() {
      return clientAccountId;
    }

    public String getMatchedHomeDomain() {
      return matchedHomeDomain;
    }

    @Override
    public int hashCode() {
      return Objects.hashCode(this.transaction.hashHex(), this.clientAccountId);
    }

    @Override
    public boolean equals(Object object) {
      if (object == this) {
        return true;
      }

      if (!(object instanceof ChallengeTransaction)) {
        return false;
      }

      ChallengeTransaction other = (ChallengeTransaction) object;
      return Objects.equal(this.transaction.hashHex(), other.transaction.hashHex()) &&
        Objects.equal(this.clientAccountId, other.clientAccountId) &&
        Objects.equal(this.matchedHomeDomain, other.matchedHomeDomain);
    }
  }

  /**
   * Represents a transaction signer.
   */
  public static class Signer {
    private final String key;
    private final int weight;

    public Signer(String key, int weight) {
      this.key = key;
      this.weight = weight;
    }

    public String getKey() {
      return key;
    }

    public int getWeight() {
      return weight;
    }

    @Override
    public int hashCode() {
      return Objects.hashCode(this.key, this.weight);
    }

    @Override
    public boolean equals(Object object) {
      if (object == this) {
        return true;
      }

      if (!(object instanceof Signer)) {
        return false;
      }

      Signer other = (Signer) object;
      return Objects.equal(this.key, other.key) &&
              Objects.equal(this.weight, other.weight);
    }
  }
}
