/*
 *  Copyright 2012 Phuong LeCong
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package com.github.plecong.hogan.groovy

import java.util.regex.Pattern

import com.github.plecong.hogan.BaseHoganTemplate
import com.github.plecong.hogan.HoganTemplate
import com.github.plecong.hogan.HoganCompiler
import com.github.plecong.hogan.TemplateLoader
import com.github.plecong.hogan.utils.ObjectUtils

/**
 * The base script used for content generated by a compiled
 * template. Provides methods to access the context and is
 * equivalent to Hogan.Template in Hogan.js (name changed
 * to match Groovy templating convention)
 */
abstract class GroovyHoganTemplate extends BaseHoganTemplate {

	static Pattern AMP_PATTERN = ~/&/
	static Pattern LT_PATTERN = ~/</
	static Pattern GT_PATTERN = ~/>/
	static Pattern APOS_PATTERN = ~/\'/
	static Pattern QUOT_PATTERN = ~/\"/
	static Pattern HCHARS_PATTERN = ~/[&<>\"\']/

	final String source
	final HoganCompiler compiler
	final Map options
	final Map partials
	final Map subs

	private StringBuilder buffer

	GroovyHoganTemplate(String source, HoganCompiler compiler, Map options = [:], Map p = [:], Map u = [:]) {
		this.source = source
		this.compiler = compiler
		this.options = options

		this.partials = (codeData.partials ?: [:]) + p
		this.subs = (codeData.subs ?: [:]) + u

		ib();
	}

	// overridden by compiled
	abstract Map<String,Map> getCodeData()
	abstract String r(Deque context, TemplateLoader partials, String indent)

	// variable escaping (was hoganEscape())
	String v(Object str) {
		str = t(str)
		return HCHARS_PATTERN.matcher(str) ?
			str
				.replaceAll(AMP_PATTERN, '&amp;')
				.replaceAll(LT_PATTERN, '&lt;')
				.replaceAll(GT_PATTERN, '&gt;')
				.replaceAll(APOS_PATTERN, '&#39;')
				.replaceAll(QUOT_PATTERN, '&quot;') :
			str
	}

	// triple stache
	String t(Object val) {
		coerceToString(val)
	}

	String coerceToString(Object val) {
		val == null ? '' : val as String
	}

	void render(Writer writer, Object context, TemplateLoader loader) {
		def contextStack = new ArrayDeque()
		contextStack.push(context ?: [:])
		writer.write(ri(contextStack, loader))
		writer.flush()
	}

	// render internal
	String ri(Deque context, TemplateLoader partials, String indent = '') {
		r(context, partials, indent)
	}

	// ensure partial
	HoganTemplate ep(String symbol, TemplateLoader partials) {
		Map<String,Object> partial = this.partials[symbol]
		String partialName = (String)partial['name']
		def template = partials[partialName]
		HoganTemplate compiledTemplate = null

		if (partial.instance && partial.base == template) {
			return partial.instance
		}

		if (template instanceof String) {
			if (!compiler) {
				throw new RuntimeException('No compiler available.')
			}
			compiledTemplate = compiler.compile(template, options + [name: getClass().name + '_' + partial.name])
		} else {
			compiledTemplate = template
		}

		if (!compiledTemplate) {
			return null
		}

		partial.base = template
		if (partial.subs) {
			// this is "specialized" because it has it's own set of subs
			// that will override the default ones compiled into the template
			// based on this execution
			compiledTemplate = createSpecializedPartial(compiledTemplate, (Map)partial.subs, (Map)partial.partials)
		}
		partial.instance = compiledTemplate
		return compiledTemplate
	}

	// render partial
	def rp(String symbol, Deque context, TemplateLoader partials, String indent) {
		HoganTemplate partial = ep(symbol, partials)
		partial ? partial.ri(context, partials, indent) : ''
	}

	// render section
	def rs(Deque context, TemplateLoader partials, Closure section) {
		def tail = context.peek()

		if (!ObjectUtils.isArray(tail)) {
			// do we need to clone here?
			Closure c = section.clone()
			c.call() // the generated code will inherit the context/partials references
			return
		}

		tail.each {
			context.push(it)
			Closure c = section.clone()
			c.call()
			context.pop()
		}
	}

	// section
	def s(Object val, Deque ctx, TemplateLoader partials, boolean inverted, int start, int end, String tags) {
		boolean pass

		// TODO: for static compilation have to check for each type
		if (ObjectUtils.isArray(val)) {
			def collection = (Collection)val
			if (collection.size() == 0) {
				return false
			}
		}

		if (val instanceof Closure) {
			val = this.ms(val, ctx, partials, inverted, start, end, tags)
		}

		// default groovy truthyness will work now that the js code is
		// (val === 0) || !!val
		pass = val == '' || val as Boolean

		if (!inverted && pass && ctx) {
			ctx.push(!ObjectUtils.isPrimitive(val) ? val : ctx.peek())
		}

		return pass
	}

	// dotted-find
	def d(String key, Deque ctx, TemplateLoader partials, boolean returnFound) {
		def names = key.tokenize('.')
		def val = f(names[0], ctx, partials, returnFound)
		def cx = null

		if (key == '.' && ObjectUtils.isArray(ctx.toArray()[-2])) {
			return ctx.peek()
		}

		for (int i = 1; i < names.size(); i++) {
			if (val && !ObjectUtils.isPrimitive(val) && ObjectUtils.hasProperty(val, names[i])) {
				cx = val
				val = val[names[i]]
			} else {
				val = ''
			}
		}

		if (returnFound && !val) {
			return false
		}

		if (!returnFound && val instanceof Closure) {
			ctx.push(cx)
			val = this.mv(val, ctx, partials)
			ctx.pop()
		}

		return val
	}

	// find
	def f(String key, Deque ctx, TemplateLoader partials, boolean returnFound) {
		def level = ctx.find { m -> ObjectUtils.hasProperty(m, key) }

		if (level == null) {
			return (returnFound) ? false : ''
		}

		def val = level[key]

		if (!returnFound && val instanceof Closure) {
			val = this.mv(val, ctx, partials)
		}

		val
	}

	// lambda section: higher order templates
	def ls(Closure val, Object cx, TemplateLoader partials, String text, String tags) {
		Closure cl = val.clone()
		cl.delegate = cx
		def result

		if (cl.maximumNumberOfParameters >= 1) {
			result = cl.call(text)
		} else {
			result = cl.call()
		}

		b(ct(coerceToString(result), cx, partials, [delimiters: tags]))
		return false
	}

	// compile template
	def ct(String text, Object cx, TemplateLoader partials, Map options = [:]) {
		if (this.options.disableLambda) {
			throw new RuntimeException('Lambda features disabled.')
		}
		compiler.compile(text, this.options + options).render(cx, partials)
	}

	// buffer
	void b(s) {
		buffer << (s ?: '')
	}

	// flush buffer
	String fl() {
		def output = buffer.toString()
		ib()
		output
	}

	// init the buffer
	void ib() {
		this.buffer = new StringBuilder()
	}

	// method replace section
	def ms(Closure func, Deque ctx, TemplateLoader partials, boolean inverted, int start, int end, String tags) {
		def cx = ctx.peek()
		Closure cl = func.clone()
		cl.delegate = cx
		def result = cl.call()

		if (result instanceof Closure) {
			if (inverted) {
				return true
			} else {
				return ls(result, cx, partials, this.source[start..<end], tags)
			}
		}

		return result
	}

	// method replace variable
	def mv(Closure func, Deque ctx, TemplateLoader partials) {
		def cx = ctx.peek()
		Closure cl = func.clone()
		cl.delegate = cx
		def result = cl.call()

		if (result instanceof Closure) {
			Closure resultCl = result.clone()
			resultCl.delegate = cx
			return ct(coerceToString(resultCl.call()), cx, partials)
		}

		return result
	}

	// sub
	def sub(String name, Deque context, TemplateLoader partials) {
		Closure cl = subs[name]

		if (cl) {
			Closure clCl = cl.clone()
			clCl.call(context, partials, this)
		}
	}

	private def createSpecializedPartial(HoganTemplate template, Map extraSubs, Map extraPartials) {
		def other = (HoganTemplate)template.class.newInstance(source, compiler, options, extraPartials, extraSubs)
		other.ib()
		other
	}

}
