package com.github.pengfeizhou.jscore;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * Created by pengfei.zhou on 2018/12/13.
 */
public class JSDecoder {
    private static final JSNull JS_NULL = new JSNull();
    private ByteBuffer byteBuffer;
    private byte[] __strBuf;

    public JSDecoder(byte[] bytes) {
        if (bytes == null || bytes.length == 0) {
            bytes = new byte[1];
            bytes[0] = 'N';
        }
        this.byteBuffer = ByteBuffer.wrap(bytes);
        byteBuffer.order(ByteOrder.BIG_ENDIAN);
    }

    /**
     * 转成bool值
     */
    public boolean bool() throws ArchiveException {
        if (!isBool()) {
            throw new ArchiveException("unable to decode bool");
        }
        return readBool();
    }

    public boolean readBool() throws ArchiveException {
        return byteBuffer.get() == 1;
    }

    /**
     * 转成string值
     */
    public String string() throws ArchiveException {
        if (!isString()) {
            throw new ArchiveException("unable to decode string");
        }
        return readString();
    }

    public String readString() throws ArchiveException {
        int len = byteBuffer.getInt();
        {
            // ceil to 4k
            int a = len / 0x1000;
            int l = (a + 1) * 0x1000;
            if (__strBuf == null || __strBuf.length < l) {
                __strBuf = new byte[l];
            }
        }
        byteBuffer.get(__strBuf, 0, len);
        try {
            return new String(__strBuf, 0, len, "utf-8");
        } catch (UnsupportedEncodingException e) {
            throw new ArchiveException("unable to decode string");
        }
    }

    /**
     * 转成number值
     */
    public Double number() throws ArchiveException {
        if (!isNumber()) {
            throw new ArchiveException("unable to decode number");
        }
        return readNumber();
    }

    public Double readNumber() {
        return byteBuffer.getDouble();
    }

    /**
     * 根据传入的解码器反解成一个java对象
     *
     * @param factory 传入指定的解码器
     */
    public <T> T object(DecodingFactory<T> factory) throws ArchiveException {
        byteBuffer.rewind();
        return readObject(factory);
    }

    public <T> T readObject(DecodingFactory<T> factory) throws ArchiveException {
        byte b = byteBuffer.get();
        if (b == 'N') {
            return factory.createInstance();
        } else if (b == 'O') {
            T obj = factory.createInstance();
            if (obj == null) {
                throw new ArchiveException("unable to create instance");
            } else if (obj instanceof Decoding) {
                ((Decoding) obj).decode(this);
                return obj;
            } else {
                throw new ArchiveException("unable to decode class: "
                        + obj.getClass().getSimpleName());
            }
        } else {
            throw new ArchiveException("unable to read object: " + this);
        }
    }

    /**
     * 根据传入的解码器反解成java对象数组
     *
     * @param factory 传入指定的解码器
     */
    public <T> T[] array(DecodingFactory<T> factory) throws ArchiveException {
        byteBuffer.rewind();
        return readArray(factory);
    }

    public <T> T[] readArray(DecodingFactory<T> factory) throws ArchiveException {
        byte b = byteBuffer.get();
        if (b == 'N') {
            return factory.createArray(0);
        }
        if (b == 'A') {
            int len = byteBuffer.getInt();
            T[] array = factory.createArray(len);
            for (int i = 0; i < len; i++) {
                array[i] = readObject(factory);
            }
            return array;
        } else {
            throw new ArchiveException("unable to read array (object): " + this);
        }
    }

    /**
     * return 0 if object is end
     */
    public int readKeyHash() throws ArchiveException {
        String name = readString();
        return name.hashCode() & 0xffff;
    }

    /**
     * 判断是否bool
     */
    public boolean isBool() {
        byteBuffer.rewind();
        return byteBuffer.get() == 'B';
    }

    /**
     * 判断是否null
     */
    public boolean isNULL() {
        byteBuffer.rewind();
        return byteBuffer.get() == 'N';
    }

    /**
     * 判断是否string
     */
    public boolean isString() {
        byteBuffer.rewind();
        return byteBuffer.get() == 'S';
    }

    /**
     * 判断是否number
     */
    public boolean isNumber() {
        byteBuffer.rewind();
        return byteBuffer.get() == 'D';
    }

    /**
     * 判断是否array
     */
    public boolean isArray() {
        byteBuffer.rewind();
        return byteBuffer.get() == 'A';
    }

    /**
     * 判断是否object
     */
    public boolean isObject() {
        byteBuffer.rewind();
        return byteBuffer.get() == 'O';
    }


    public JSValue decode() throws ArchiveException {
        byte b = byteBuffer.get();
        switch (b) {
            case 'A': {
                int len = byteBuffer.getInt();
                JSArray ret = new JSArray(len);
                for (int i = 0; i < len; i++) {
                    ret.put(i, decode());
                }
                return ret;
            }
            case 'S': {
                return new JSString(readString());
            }
            case 'D': {
                return new JSNumber(readNumber());
            }
            case 'B': {
                return new JSBoolean(readBool());
            }
            case 'O': {
                JSObject jsObject = new JSObject();
                JSValue propertyName;
                while ((propertyName = decode()) instanceof JSString) {
                    jsObject.setProperty(((JSString) propertyName).value(), decode());
                }
                return jsObject;
            }
            case 'N':
            default:
                return JS_NULL;
        }
    }
}
