package com.nedap.archie.json.flat;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nedap.archie.aom.OperationalTemplate;
import com.nedap.archie.base.OpenEHRBase;
import com.nedap.archie.creation.ExampleJsonInstanceGenerator;
import com.nedap.archie.rminfo.MetaModels;

import java.util.Map;

public class FlatJsonExampleInstanceGenerator {

    /**
     * Generate a flat JSON example instance for the given OperationalTemplate.
     *
     * @param template the template for which to generate the example
     * @param metaModels the metamodels to use. Must contain an actual RM ModelInfoLookup and a JSON Object Mapper
     * @param language the language to generate the example in
     * @param jsonFormatConfiguration the configuratin of the flat format
     * @return a flat json as generated by the FlatJsonGenerator, to be serialized with an objectmapper
     * @throws JsonProcessingException in case the ExampleJsonInstanceGenerator generates json that the objectmapper cannot process
     * @throws DuplicateKeyException in case the FlatJsonGenerator generates incorrect data for this OperationalTemplate
     */
    public Map<String, Object> generateExample(OperationalTemplate template, MetaModels metaModels, String language, FlatJsonFormatConfiguration jsonFormatConfiguration) throws JsonProcessingException, DuplicateKeyException {
        metaModels.selectModel(template);
        if(metaModels.getSelectedModel() == null) {
            throw new IllegalArgumentException("Cannot find MetaModel for archetype");
        }
        if(metaModels.getSelectedModelInfoLookup() == null) {
            throw new IllegalArgumentException("Cannot find ModelInfoLookup for archetype");
        }
        if(metaModels.getSelectedModel().getJsonObjectMapper() == null) {
            throw new IllegalArgumentException("Cannot find JSON Object mapper in selected metamodel");
        }
        ExampleJsonInstanceGenerator exampleJsonInstanceGenerator = new ExampleJsonInstanceGenerator(metaModels, language);
        exampleJsonInstanceGenerator.setTypePropertyName("_type");
        Map<String, Object> generatedExample = exampleJsonInstanceGenerator.generate(template);
        ObjectMapper objectMapper = metaModels.getSelectedModel().getJsonObjectMapper();
        String jsonRmObject = objectMapper.writeValueAsString(generatedExample);
        OpenEHRBase openEHRBase = objectMapper.readValue(jsonRmObject, OpenEHRBase.class);

        return new FlatJsonGenerator(metaModels.getSelectedModelInfoLookup(), jsonFormatConfiguration).buildPathsAndValues(openEHRBase);

    }
}
