package io.github.luizgrp.sectionedrecyclerviewadapter;

import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;

/**
 * A custom RecyclerView with Sections with custom Titles.
 * Sections are displayed in the same order they were added.
 *
 * @author Gustavo Pagani
 */
public class SectionedRecyclerViewAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> {

    private LinkedHashMap<String, Section> sections;

    public SectionedRecyclerViewAdapter() {
        sections = new LinkedHashMap<>();
    }

    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        RecyclerView.ViewHolder viewHolder = null;
        View view = null;

        int loopViewType = 0;

        for (Map.Entry<String, Section> entry : sections.entrySet()) {
            Section section = entry.getValue();

            // ignore invisible sections
            if (!section.isVisible()) continue;

            if (loopViewType == viewType) {

                Integer resId = section.getHeaderResourceId();

                if (resId == null) throw new NullPointerException("Missing 'header' resource id");

                view = LayoutInflater.from(parent.getContext()).inflate(resId, parent, false);
                // get the header viewholder from the section
                viewHolder = section.getHeaderViewHolder(view);
                break;
            }

            loopViewType++;

            if (loopViewType == viewType) {

                Integer resId = section.getFooterResourceId();

                if (resId == null) throw new NullPointerException("Missing 'footer' resource id");

                view = LayoutInflater.from(parent.getContext()).inflate(resId, parent, false);
                // get the footer viewholder from the section
                viewHolder = section.getFooterViewHolder(view);
                break;
            }

            loopViewType++;

            if (loopViewType == viewType) {
                view = LayoutInflater.from(parent.getContext()).inflate(section.getItemResourceId(), parent, false);
                // get the item viewholder from the section
                viewHolder = section.getItemViewHolder(view);
                break;
            }

            loopViewType++;

            if (loopViewType == viewType) {
                Integer resId = section.getLoadingResourceId();

                if (resId == null) throw new NullPointerException("Missing 'loading state' resource id");

                view = LayoutInflater.from(parent.getContext()).inflate(resId, parent, false);
                // get the loading viewholder from the section
                viewHolder = section.getLoadingViewHolder(view);
                break;
            }

            loopViewType++;

            if (loopViewType == viewType) {
                Integer resId = section.getFailedResourceId();

                if (resId == null) throw new NullPointerException("Missing 'failed state' resource id");

                view = LayoutInflater.from(parent.getContext()).inflate(resId, parent, false);
                // get the failed load viewholder from the section
                viewHolder = section.getFailedViewHolder(view);
                break;
            }

            loopViewType++;
        }

        if (viewHolder == null) throw new IllegalArgumentException("Invalid viewType");

        return viewHolder;
    }

    /**
     * Add a section to this recyclerview.
     * @param tag unique identifier of the section
     * @param section section to be added
     */
    public void addSection(String tag, Section section) {
        this.sections.put(tag, section);
    }

    /**
     * Add a section to this recyclerview with a random tag;
     * @param section section to be added
     * @return generated tag
     */
    public String addSection(Section section) {
        String tag = UUID.randomUUID().toString();

        addSection(tag, section);

        return tag;
    }

    /**
     * Return the section with the tag provided
     * @param tag unique identifier of the section
     * @return section
     */
    public Section getSection(String tag) {
        return this.sections.get(tag);
    }

    /**
     * Remove section from this recyclerview.
     * @param tag unique identifier of the section
     */
    public void removeSection(String tag) {
        this.sections.remove(tag);
    }

    /**
     * Remove all sections from this recyclerview.
     */
    public void removeAllSections() {
        this.sections.clear();
    }

    @Override
    public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {

        int currentPos = 0;

        for (Map.Entry<String, Section> entry : sections.entrySet()) {
            Section section = entry.getValue();

            // ignore invisible sections
            if (!section.isVisible()) continue;

            int sectionTotal = section.getSectionItemsTotal();

            // check if position is in this section
            if (position >= currentPos && position <= (currentPos + sectionTotal - 1)) {

                if (section.hasHeader()) {
                    if (position == currentPos) {
                        // delegate the binding to the section header
                        getSectionForPosition(position).onBindHeaderViewHolder(holder);
                        return;
                    }
                }

                if (section.hasFooter()) {
                    if (position == (currentPos + sectionTotal - 1)) {
                        // delegate the binding to the section header
                        getSectionForPosition(position).onBindFooterViewHolder(holder);
                        return;
                    }
                }

                // delegate the binding to the section content
                getSectionForPosition(position).onBindContentViewHolder(holder, getSectionPosition(position));
                return;
            }

            currentPos += sectionTotal;
        }

        throw new IndexOutOfBoundsException("Invalid position");
    }

    @Override
    public int getItemCount() {
        int count = 0;

        for (Map.Entry<String, Section> entry : sections.entrySet()) {
            Section section = entry.getValue();

            // ignore invisible sections
            if (!section.isVisible()) continue;

            count += section.getSectionItemsTotal();
        }

        return count;
    }

    @Override
    public int getItemViewType(int position) {
        /*
         Each Section has 5 "viewtypes":
         1) header
         2) footer
         3) items
         4) loading
         5) load failed
         */
        int viewType = 0;
        int currentPos = 0;

        for (Map.Entry<String, Section> entry : sections.entrySet()) {
            Section section = entry.getValue();

            // ignore invisible sections
            if (!section.isVisible()) continue;

            int sectionTotal = section.getSectionItemsTotal();

            // check if position is in this section
            if (position >= currentPos && position <= (currentPos + sectionTotal - 1)) {

                if (section.hasHeader()) {
                    if (position == currentPos) {
                        return viewType;
                    }
                }

                if (section.hasFooter()) {
                    if (position == (currentPos + sectionTotal - 1)) {
                        return viewType + 1;
                    }
                }

                switch (section.getState()) {
                    case LOADED:
                        return viewType + 2;
                    case LOADING:
                        return viewType + 3;
                    case FAILED:
                        return viewType + 4;
                    default:
                        throw new IllegalStateException("Invalid state");
                }

            }

            currentPos += sectionTotal;

            viewType += 5;
        }

        throw new IndexOutOfBoundsException("Invalid position");
    }

    /**
     * Returns the Section object for that position
     * @param position position in the recyclerview
     * @return Section object for that position
     */
    Section getSectionForPosition(int position) {

        int currentPos = 0;

        for (Map.Entry<String, Section> entry : sections.entrySet()) {
            Section section = entry.getValue();

            // ignore invisible sections
            if (!section.isVisible()) continue;

            int sectionTotal = section.getSectionItemsTotal();

            // check if position is in this section
            if (position >= currentPos && position <= (currentPos + sectionTotal - 1)) {
                return section;
            }

            currentPos += sectionTotal;
        }

        throw new IndexOutOfBoundsException("Invalid position");
    }

    /**
     * Return the item position relative to the section.
     * @param position position of the item in the recyclerview
     * @return position of the item in the section
     */
    int getSectionPosition(int position) {
        int currentPos = 0;

        for (Map.Entry<String, Section> entry : sections.entrySet()) {
            Section section = entry.getValue();

            // ignore invisible sections
            if (!section.isVisible()) continue;

            int sectionTotal = section.getSectionItemsTotal();

            // check if position is in this section
            if (position >= currentPos && position <= (currentPos + sectionTotal - 1)) {
                return position - currentPos - (section.hasHeader() ? 1 : 0);
            }

            currentPos += sectionTotal;
        }

        throw new IndexOutOfBoundsException("Invalid position");
    }

    /**
     * A concrete class of an empty ViewHolder.
     * Should be used to avoid the boilerplate of creating a ViewHolder class for simple case
     * scenarios.
     */
    public static class EmptyViewHolder extends RecyclerView.ViewHolder {
        public EmptyViewHolder(View itemView) {
            super(itemView);
        }
    }
}
