package in.kyle.api.generate.processors.copy;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtField;
import javassist.CtMethod;
import javassist.CtNewMethod;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;

import in.kyle.api.generate.api.Generated;
import in.kyle.api.generate.api.GeneratedFunction;
import in.kyle.api.generate.api.GeneratorHandler;
import in.kyle.api.generate.api.Invoke;
import in.kyle.api.generate.api.Matcher;
import in.kyle.api.generate.helper.Assist;
import in.kyle.api.generate.helper.ReflectHelper;
import in.kyle.api.generate.processors.Processor;
import in.kyle.api.utils.StringUtils;
import in.kyle.api.utils.Try;

public class CopyProcessor implements Processor<Method> {
    
    private static final ClassPool pool = Assist.POOL;
    static final Map<String, Class<?>> genClasses = new HashMap<>();
    
    @SuppressWarnings("unchecked")
    @Override
    public <I extends Generated> void process(Class<I> clazz,
                                              I instance,
                                              Method srcMethod,
                                              GeneratorHandler handler) throws Exception {
        
        Copy copy = srcMethod.getAnnotation(Copy.class);
        Class<?> srcClass = copy.value();
        Class target;
        synchronized (genClasses) {
            target = Try.to(() -> makeImplementingClass(srcMethod, srcClass));
        }
        
        handler.addFunction(new Function(target, srcMethod));
    }
    
    private Class<?> makeImplementingClass(Method srcMethod, Class<?> srcClass) throws Exception {
        String name = StringUtils.replaceVariables("Gen_Target_{}_{}",
                                                   srcMethod.getDeclaringClass()
                                                            .getName()
                                                            .replace(".", "_"),
                                                   srcMethod.getName());
        if (genClasses.containsKey(name)) {
            return genClasses.get(name);
        }
    
        CtClass ctTarget = pool.makeClass(name);
        ctTarget.setSuperclass(pool.get(srcMethod.getDeclaringClass().getName()));
        CtClass ctSrc = Assist.getClass(srcClass);
        CtClass[] invokeParams = Assist.getClasses(srcMethod.getParameterTypes());
        CtMethod ctSrcMethod = ctSrc.getDeclaredMethod(srcMethod.getName(), invokeParams);
        CtMethod targetMethod = CtNewMethod.copy(ctSrcMethod, ctTarget, null);
        ctTarget.addMethod(targetMethod);
        
        CtClass invokeClass = pool.get(srcMethod.getDeclaringClass().getName());
        for (CtField field : invokeClass.getDeclaredFields()) {
            CtField ctSrcField = new CtField(field, ctTarget);
            ctSrcField.setModifiers(Modifier.PUBLIC);
            ctTarget.addField(ctSrcField);
        }
        
        Class target = ctTarget.toClass();
        genClasses.put(name, target);
        return target;
    }
    
    static class Function implements GeneratedFunction {
        
        private final Matcher matcher;
        private final Class<? extends Generated> target;
        private final Method copiedMethod;
        
        Function(Class<? extends Generated> target, Method method) throws NoSuchMethodException {
            this.target = target;
            this.matcher = Matcher.exact(method);
            this.copiedMethod =
                    target.getDeclaredMethod(method.getName(), method.getParameterTypes());
        }
        
        @Override
        public Matcher getMatcher() {
            return matcher;
        }
        
        @Override
        public Object invoke(GeneratorHandler handler, Invoke invoke) throws Throwable {
            Object generatedInstance = handler.getGenerator().create(target);
            
            copyFieldValues(generatedInstance, handler);
            return copiedMethod.invoke(generatedInstance, invoke.getArgs());
        }
        
        private void copyFieldValues(Object to, GeneratorHandler handler)
                throws IllegalAccessException {
            for (Field field : ReflectHelper.getFields(to.getClass())) {
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                field.setAccessible(true);
                field.set(to, handler.getValue(field.getName()));
            }
        }
    }
}
