package in.kyle.api.generate.processors;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.Collection;

import in.kyle.api.generate.api.GeneratorHandler;
import in.kyle.api.generate.api.Generated;
import in.kyle.api.generate.api.Matcher;
import in.kyle.api.generate.helper.CamelCaseHelper;
import lombok.AllArgsConstructor;

@AllArgsConstructor
public abstract class CollectionProcessor<A extends Annotation, B extends Collection<Object>>
        implements Processor<Field> {
    
    private final Class<A> annotationClass;
    
    @Override
    public <I extends Generated> void process(Class<I> clazz,
                                              I instance,
                                              Field field,
                                              GeneratorHandler handler) throws Exception {
        A annotation = field.getAnnotation(annotationClass);
        
        String methodSuffix = CamelCaseHelper.toUpperCamelCase(field.getName());
        methodSuffix = methodSuffix.substring(0, methodSuffix.length() - 1); // remove plural
        B value = handler.getValueOrDefault(field.getName(), createDefaultCollection());
        
        Class<?> type = getType(annotation);
        
        // add
        Matcher addMatcher = Matcher.builder().args(type).nameEquals("add" + methodSuffix).build();
        handler.createAndAddFunction(addMatcher, invoke -> value.add(invoke.getArg(0)));
        
        // add force
        Matcher addForceMatcher =
                Matcher.builder().args(type, Boolean.TYPE).nameEquals("add" + methodSuffix).build();
        handler.createAndAddFunction(addForceMatcher, invoke -> value.add(invoke.getArg(0)));
        
        // add plural
        Matcher addPluralMatcher = Matcher.builder()
                                          .args(Collection.class)
                                          .nameEquals("add" + methodSuffix + "s")
                                          .build();
        handler.createAndAddFunction(addPluralMatcher, invoke -> value.addAll(invoke.getArg(0)));
        
        // has
        Matcher hasMater = Matcher.builder().args(type).nameEquals("has" + methodSuffix).build();
        handler.createAndAddFunction(hasMater, invoke -> value.contains(invoke.getArg(0)));
        
        // remove
        Matcher removeMater =
                Matcher.builder().args(type).nameEquals("remove" + methodSuffix).build();
        handler.createAndAddFunction(removeMater, invoke -> value.remove(invoke.getArg(0)));
    }
    
    protected abstract Class<?> getType(A annotation);
    
    protected abstract B createDefaultCollection();
}
