package in.kyle.api.generate.api;

import javassist.util.proxy.MethodHandler;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import in.kyle.api.generate.DefaultFunctions;
import in.kyle.api.generate.NotImplementedException;
import in.kyle.api.generate.helper.ReflectHelper;
import in.kyle.api.utils.Try;
import in.kyle.api.verify.Verify;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;

public class GeneratorHandler implements MethodHandler {
    
    @Getter
    protected final List<GeneratedFunction> functions = new ArrayList<>();
    private final Map<String, Object> values = new HashMap<>();
    @Getter
    private final Generator generator;
    
    @Setter(AccessLevel.PACKAGE)
    private Object object;
    
    public GeneratorHandler(Generator generator) {
        this.generator = generator;
        DefaultFunctions.addTo(this);
    }
    
    public void addFunction(GeneratedFunction function) {
        functions.add(function);
    }
    
    public <T> T getValueOrDefault(String name, T def) {
        Verify.that(name).isNotNull();
        Object value = getValue(name);
        if (value != null) {
            return (T) value;
        } else {
            setValue(name, def);
            return def;
        }
    }
    
    public Object getValue(String name) {
        Object temp = getFieldValue(name);
        if (temp == null) {
            temp = values.get(name);
        }
        return temp;
    }
    
    public Object getFieldValue(String name) {
        Field field = ReflectHelper.getAccessibleField(object.getClass(), name);
        if (field != null) {
            return Try.to(() -> field.get(object));
        } else {
            return null;
        }
    }
    
    public Object getDefaultReturnValue(Method method) {
        Class<?> returnType = method.getReturnType();
        return Defaults.defaultValue(returnType);
    }
    
    public void setValue(String name, Object value) {
        Field field = ReflectHelper.getAccessibleField(object.getClass(), name);
        if (field == null) {
            values.put(name, value);
        } else {
            Try.to(() -> field.set(object, value));
        }
    }
    
    @Override
    public Object invoke(Object o, Method method, Method proxyMethod, Object[] args)
            throws Throwable {
        Invoke invoke = new Invoke(method, o, args == null ? new Object[0] : args);
        GeneratedFunction function = functions.stream()
                                              .filter(e -> e.getMatcher().matches(invoke))
                                              .findFirst()
                                              .orElseThrow(() -> {
                                                  return new NotImplementedException(method.toString());
                                              });
        return function.invoke(this, invoke);
    }
    
    public Map<String, Object> getEffectiveValues() throws IllegalAccessException {
        Map<String, Object> effective = new HashMap<>(values);
        for (Field field : ReflectHelper.getFields(object.getClass())) {
            field.setAccessible(true);
            if (field.get(object) != null && !Modifier.isStatic(field.getModifiers())) {
                effective.put(field.getName(), field.get(object));
            }
        }
        return Collections.unmodifiableMap(effective);
    }
    
    public void createAndAddFunction(Matcher matcher, Function<Invoke, Object> function) {
        addFunction(createFunction(matcher, function));
    }
    
    public static GeneratedFunction createFunction(Matcher matcher,
                                                   Function<Invoke, Object> function) {
        return new GeneratedFunction() {
            @Override
            public Matcher getMatcher() {
                return matcher;
            }
            
            @Override
            public Object invoke(GeneratorHandler handler, Invoke invoke) throws Throwable {
                return function.apply(invoke);
            }
        };
    }
}