package in.kyle.api.generate.api;

import javassist.util.proxy.ProxyFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import in.kyle.api.generate.helper.ReflectHelper;
import in.kyle.api.generate.processors.Global;
import in.kyle.api.generate.processors.Processor;
import in.kyle.api.generate.processors.collection.list.ImplementList;
import in.kyle.api.generate.processors.collection.list.ListProcessor;
import in.kyle.api.generate.processors.collection.set.ImplementSet;
import in.kyle.api.generate.processors.collection.set.SetProcessor;
import in.kyle.api.generate.processors.copy.Copy;
import in.kyle.api.generate.processors.copy.CopyProcessor;
import in.kyle.api.generate.processors.create.Create;
import in.kyle.api.generate.processors.create.CreateProcessor;
import in.kyle.api.generate.processors.increment.Increment;
import in.kyle.api.generate.processors.increment.IncrementProcessor;
import in.kyle.api.generate.processors.map.ImplementMap;
import in.kyle.api.generate.processors.map.MapProcessor;
import in.kyle.api.generate.processors.setup.Setup;
import in.kyle.api.generate.processors.setup.SetupProcessor;
import in.kyle.api.utils.Try;

@SuppressWarnings("unchecked")
public final class Generator {
    
    private static final Map<Class<?>, Object> globals = new HashMap<>();
    
    protected final Map<Class<? extends Annotation>, Processor<?>> processors;
    
    private Generator() {
        processors = new LinkedHashMap<>();
        addDefaultProcessors();
    }
    
    private void addDefaultProcessors() {
        addProcessor(Create.class, new CreateProcessor());
        addProcessor(Copy.class, new CopyProcessor());
        addProcessor(Increment.class, new IncrementProcessor());
        addProcessor(ImplementSet.class, new SetProcessor());
        addProcessor(ImplementList.class, new ListProcessor());
        addProcessor(ImplementMap.class, new MapProcessor());
        addProcessor(Setup.class, new SetupProcessor());
    }
    
    public void inject(Object target) {
        ReflectHelper.getFields(target.getClass())
                     .stream()
                     .filter(field -> Generated.class.isAssignableFrom(field.getType()))
                     .forEach(field -> {
                         field.setAccessible(true);
            
                         Try.to(() -> inject(target, field),
                                e -> new GenerateException(e,
                                                           "Unable to inject field {}:{} in " +
                                                           "class {}",
                                                           field.getType().getSimpleName(),
                                                           field.getName(),
                                                           field.getDeclaringClass()));
                     });
    }
    
    public synchronized <T extends Generated> T create(Class<T> clazz) {
        synchronized (globals) {
            if (globals.containsKey(clazz)) {
                return (T) globals.get(clazz);
            }
        }
        GeneratorHandler handler = new GeneratorHandler(this);
        T instance = createInstance(clazz, handler);
        modifyMembers(clazz);
        
        Collection<AnnotatedElement> members = ReflectHelper.getAnnotatedElements(clazz);
        processors.forEach((a, p) -> {
            for (AnnotatedElement member : members) {
                if (member.isAnnotationPresent(a)) {
                    Try.to(() -> ((Processor) p).process(clazz, instance, member, handler),
                           e -> new GenerateException(e, "Unable to process element {}", member));
                }
            }
        });
        
        processGlobals(clazz, instance);
        
        return instance;
    }
    
    private <T extends Generated> void processGlobals(Class<T> clazz, T instance) {
        for (Field field : ReflectHelper.getFields(clazz)) {
            field.setAccessible(true);
            Try.to(() -> {
                       if (field.getType().isAnnotationPresent(Global.class) &&
                           field.get(instance) == null) {
                           Class<? extends Generated> type = (Class<? extends Generated>) field
                                   .getType();
                           field.set(instance, create(type));
                       }
                   },
                   e -> new GenerateException(e,
                                              "Failed to inject global variable {} in {}",
                                              field.getName(),
                                              clazz.getName()));
        }
    }
    
    private <T extends Generated> void modifyMembers(Class<T> clazz) {
        ReflectHelper.getMembers(clazz)
                     .stream()
                     .map(m -> (AccessibleObject) m)
                     .forEach(a -> a.setAccessible(true));
        
        for (Field field : ReflectHelper.getFields(clazz)) {
            field.setAccessible(true);
        }
    }
    
    private <T extends Generated> T createInstance(Class<T> clazz, GeneratorHandler handler)
            throws GenerateException {
        if (!Generated.class.isAssignableFrom(clazz)) {
            throw new GenerateException("Not generated instance {}", clazz.getName());
        }
        boolean global = clazz.isAnnotationPresent(Global.class);
        
        ProxyFactory factory = new ProxyFactory();
        factory.setSuperclass(clazz);
        factory.setFilter(method -> Modifier.isAbstract(method.getModifiers()));
        
        Constructor<?> constructor = clazz.getDeclaredConstructors()[0];
        
        Object[] values = new Object[constructor.getParameterCount()];
        for (int i = 0; i < values.length; i++) {
            values[i] = create((Class<T>) constructor.getParameterTypes()[i]);
        }
        T t = (T) Try.to(() -> factory.create(constructor.getParameterTypes(), values, handler),
                         e -> makeGenerateException(clazz, e));
        handler.setObject(t);
        
        if (global) {
            synchronized (globals) {
                globals.put(clazz, t);
            }
        }
        
        return t;
    }
    
    private <T extends Generated> Throwable makeGenerateException(Class<T> clazz, Throwable e) {
        return new GenerateException(e, "Failed to create instance of {}", clazz.getName());
    }
    
    public <T extends Annotation> void addProcessor(Class<T> annotation, Processor<?> processor) {
        processors.put(annotation, processor);
    }
    
    private void inject(Object target, Field field) throws IllegalAccessException {
        field.set(target, create((Class<? extends Generated>) field.getType()));
    }
    
    public static Generator create() {
        return new Generator();
    }
}
