package in.kyle.api.bukkit;


import org.bukkit.Chunk;
import org.bukkit.Difficulty;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.WorldType;
import org.bukkit.block.Block;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Item;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Player;
import org.bukkit.generator.ChunkGenerator;
import org.bukkit.inventory.ItemStack;
import org.bukkit.util.Consumer;
import org.bukkit.util.Vector;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import in.kyle.api.bukkit.entity.EntityFactory;
import in.kyle.api.bukkit.entity.TestItem;
import in.kyle.api.generate.api.Generated;
import in.kyle.api.utils.Conditions;
import lombok.Data;

@Data
public abstract class TestWorld implements World, Generated {
    
    private static final AtomicInteger idIndex = new AtomicInteger();
    
    private final TestServer server;
    
    private final Map<Long, Chunk> chunks = new HashMap<>();
    private final Map<Vector, Entity> entities = new HashMap<>();
    private final UUID uid = UUID.randomUUID();
    private final Map<String, String> gameRules = new HashMap<>();
    
    
    private Location spawnLocation = new Location(this, 0, 0, 0);
    
    private String name = "world-" + idIndex.incrementAndGet();
    private long time = 0;
    private long fullTime = 0;
    private boolean pvp = true;
    private Environment environment = Environment.NORMAL;
    private ChunkGenerator generator = new TestChunkGenerator();
    
    private boolean storm = false;
    private boolean thundering;
    private int thunderDuration = 0;
    private int weatherDuration = 0;
    private boolean allowAnimals = true;
    private boolean allowMonsters = true;
    private Difficulty difficulty = Difficulty.EASY;
    
    public TestWorld(TestServer server) {
        this.server = server;
    }
    
    private long getKey(Chunk chunk) {
        return getKey(chunk.getX(), chunk.getZ());
    }
    
    private long getKey(int chunkX, int chunkY) { // good enough
        return (long) chunkX << 31 | chunkY;
    }
    
    private long getChunk(int blockX, int blockZ) {
        return getKey(blockX >> 4, blockZ >> 4);
    }
    
    public Block getBlockAt(int x, int y, int z) {
        return chunks.get(getChunk(x, z)).getBlock(x, y, z);
    }
    
    public Block getBlockAt(Location location) {
        Conditions.isTrue(equals(location.getWorld()), "Worlds differ");
        return getBlockAt(location.getBlockX(), location.getBlockY(), location.getBlockZ());
    }
    
    public int getBlockTypeIdAt(int x, int y, int z) {
        return getBlockAt(x, y, z).getTypeId();
    }
    
    public int getBlockTypeIdAt(Location location) {
        return getBlockAt(location).getTypeId();
    }
    
    public Chunk getChunkAt(int x, int z) {
        return chunks.get(getKey(x, z));
    }
    
    public Chunk getChunkAt(Location location) {
        Conditions.isTrue(equals(location.getWorld()), "Worlds differ");
        return getChunkAt(location.getBlockX(), location.getBlockZ());
    }
    
    public Chunk getChunkAt(Block block) {
        return getChunkAt(block.getLocation());
    }
    
    public boolean isChunkLoaded(Chunk chunk) {
        return chunks.containsKey(getKey(chunk.getX(), chunk.getZ()));
    }
    
    public Chunk[] getLoadedChunks() {
        return chunks.values().toArray(new Chunk[chunks.size()]);
    }
    
    public void loadChunk(Chunk chunk) {
        chunks.put(getKey(chunk), chunk);
    }
    
    public boolean isChunkLoaded(int x, int z) {
        return getChunkAt(x, z) != null;
    }
    
    public boolean unloadChunk(Chunk chunk) {
        return chunks.remove(getKey(chunk)) != null;
    }
    
    public boolean unloadChunk(int x, int y) {
        return chunks.remove(getKey(x, y)) != null;
    }
    
    public boolean unloadChunk(int x, int y, boolean save) {
        return unloadChunk(x, y);
    }
    
    public boolean unloadChunk(int x, int y, boolean save, boolean safe) {
        return unloadChunk(x, y, save);
    }
    
    public boolean unloadChunkRequest(int x, int y) {
        return unloadChunk(x, y);
    }
    
    public boolean unloadChunkRequest(int x, int y, boolean safe) {
        return unloadChunk(x, y);
    }
    
    public boolean regenerateChunk(int x, int y) {
        unloadChunk(x, y);
        TestChunk chunk = ((Generated) this).getInternalGenerator().create(TestChunk.class);
        chunk.setX(x);
        chunk.setZ(y);
        chunk.setWorld(this);
        loadChunk(chunk);
        return true;
    }
    
    public boolean refreshChunk(int x, int y) {
        return true;
    }
    
    public void removeEntity(Entity entity) {
        entities.remove(entity.getLocation().toVector(), entity);
    }
    
    public Item dropItem(Location location, ItemStack itemStack) {
        TestItem testItem = ((Generated) this).getInternalGenerator().create(TestItem.class);
        testItem.setLocation(location);
        testItem.setItemStack(itemStack);
        entities.put(location.toVector(), testItem);
        return testItem;
    }
    
    public Item dropItemNaturally(Location location, ItemStack itemStack) {
        return dropItem(location, itemStack);
    }
    
    public List<Entity> getEntities() {
        return new ArrayList<>(entities.values());
    }
    
    public List<LivingEntity> getLivingEntities() {
        return new ArrayList<>(getEntitiesByClass(LivingEntity.class));
    }
    
    public <T extends Entity> Collection<T> getEntitiesByClass(Class<T>[] classes) {
        return new ArrayList<>(getEntities(classes));
    }
    
    @SafeVarargs
    private final <T extends Entity> Collection<T> getEntities(Class<T>... classes) {
        return entities.values().stream().filter(entity -> {
            for (Class<T> clazz : classes) {
                if (clazz.isAssignableFrom(entity.getClass())) {
                    return true;
                }
            }
            return false;
        }).map(e -> (T) e).collect(Collectors.toList());
    }
    
    public <T extends Entity> Collection<T> getEntitiesByClass(Class<T> aClass) {
        return getEntities(aClass);
    }
    
    public Collection<Entity> getEntitiesByClasses(Class<?>[] classes) {
        return getEntities((Class[]) classes);
    }
    
    public List<Player> getPlayers() {
        List<Player> online = new ArrayList<>();
        for (Player player : server.getOnlinePlayers()) {
            if (player.getWorld().equals(this)) {
                online.add(player);
            }
        }
        return online;
    }
    
    @Override
    public UUID getUID() {
        return uid;
    }
    
    public boolean setSpawnLocation(int x, int y, int z) {
        spawnLocation.setX(x);
        spawnLocation.setY(y);
        spawnLocation.setZ(z);
        return true;
    }
    
    public boolean hasStorm() {
        return storm;
    }
    
    public long getSeed() {
        return 0;
    }
    
    public boolean getPVP() {
        return pvp;
    }
    
    public void setPVP(boolean b) {
        pvp = b;
    }
    
    public <T extends Entity> T spawn(Location location, Class<T> clazz)
            throws IllegalArgumentException {
        T entity = EntityFactory.make(this, clazz);
        entity.teleport(location);
        entities.put(entity.getLocation().toVector(), entity);
        return entity;
    }
    
    public <T extends Entity> T spawn(Location location, Class<T> clazz, Consumer<T> consumer)
            throws IllegalArgumentException {
        T entity = spawn(location, clazz);
        consumer.accept(entity);
        return entity;
    }
    
    @Override
    public boolean getAllowAnimals() {
        return allowAnimals;
    }
    
    @Override
    public boolean getAllowMonsters() {
        return allowMonsters;
    }
    
    public int getMaxHeight() {
        return 256;
    }
    
    public int getSeaLevel() {
        return 63;
    }
    
    public WorldType getWorldType() {
        return WorldType.CUSTOMIZED;
    }
    
    public String[] getGameRules() {
        return gameRules.keySet().toArray(new String[gameRules.size()]);
    }
    
    public String getGameRuleValue(String s) {
        return gameRules.get(s);
    }
    
    public boolean setGameRuleValue(String key, String value) {
        gameRules.put(key, value);
        return true;
    }
    
    public boolean isGameRule(String key) {
        return gameRules.containsKey(key);
    }
}
