package in.kyle.api.bukkit;

import org.bukkit.Bukkit;
import org.bukkit.GameMode;
import org.bukkit.OfflinePlayer;
import org.bukkit.Server;
import org.bukkit.Warning;
import org.bukkit.World;
import org.bukkit.WorldCreator;
import org.bukkit.command.CommandException;
import org.bukkit.command.CommandSender;
import org.bukkit.command.PluginCommand;
import org.bukkit.command.SimpleCommandMap;
import org.bukkit.craftbukkit.v1_12_R1.CraftServer;
import org.bukkit.craftbukkit.v1_12_R1.scheduler.CraftScheduler;
import org.bukkit.entity.Player;
import org.bukkit.event.inventory.InventoryType;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.InventoryHolder;
import org.bukkit.permissions.Permissible;
import org.bukkit.permissions.Permission;
import org.bukkit.plugin.PluginManager;
import org.bukkit.plugin.ServicesManager;
import org.bukkit.plugin.SimpleServicesManager;
import org.bukkit.scheduler.BukkitScheduler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import in.kyle.api.bukkit.inventory.TestInventory;
import in.kyle.api.bukkit.plugin.TestPluginManager;
import in.kyle.api.generate.api.Generated;
import in.kyle.api.generate.processors.Global;
import in.kyle.api.generate.processors.copy.Copy;
import in.kyle.api.generate.processors.increment.Increment;
import in.kyle.api.utils.Try;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;

@EqualsAndHashCode(callSuper = true)
@Global
@Data
@NoArgsConstructor
public abstract class TestServer extends TestPluginManager
        implements Server, PluginManager, Generated {
    
    private final Set<Player> onlinePlayers = new HashSet<>();
    private final Set<OfflinePlayer> whitelistedPlayers = new HashSet<>();
    private final Set<OfflinePlayer> bannedPlayers = new HashSet<>();
    private final Set<OfflinePlayer> operators = new HashSet<>();
    
    private final PluginManager pluginManager = this;
    private final CraftScheduler scheduler = null;
    private final ServicesManager servicesManager = new SimpleServicesManager();
    private final List<World> worlds = new ArrayList<>();
    private final SimpleCommandMap commandMap = new SimpleCommandMap(this);
    private final Set<String> iPBans = new HashSet<>();
    
    private GameMode defaultGameMode = GameMode.SURVIVAL;
    
    @Increment
    private String serverId;
    private String name = "Test Server";
    private String serverName = name;
    private String version = "1";
    private String bukkitVersion = "1";
    private String worldType = "void";
    private String motd = "default motd";
    private String shutdownMessage = "Server shut down";
    
    private String ip = "localhost";
    private int port = 25565;
    
    private boolean schedulerStarted;
    
    {
        Bukkit.setServer(this);
    }
    
    public World createDefaultWorld() {
        worlds.add(getInternalGenerator().create(TestWorld.class));
        return getWorld();
    }
    
    public World getWorld() {
        if (worlds.size() == 0) {
            return createDefaultWorld();
        } else {
            return worlds.get(0);
        }
    }
    
    public void startScheduler(AtomicBoolean running) {
        schedulerStarted = true;
        int tick = 1;
        while (running.get()) {
            scheduler.mainThreadHeartbeat(tick);
            if (++tick > 20) {
                tick = 1;
            }
            Try.to(() -> Thread.sleep(50));
        }
    }
    
    @Override
    @Copy(CraftServer.class)
    public abstract int broadcast(String message, String permission);
    
    @Override
    @Copy(CraftServer.class)
    public abstract int broadcastMessage(String message);
    
    @Override
    @Copy(CraftServer.class)
    public abstract boolean dispatchCommand(CommandSender sender, String commandLine)
            throws CommandException;
    
    @Override
    public Player getPlayer(String name) {
        return onlinePlayers.stream()
                            .filter(player -> player.getName().equals(name))
                            .findAny()
                            .orElse(null);
    }
    
    @Override
    public Player getPlayerExact(String name) {
        return getPlayer(name);
    }
    
    @Override
    public List<Player> matchPlayer(String name) {
        return onlinePlayers.stream()
                            .filter(player -> player.getName().contains(name))
                            .collect(Collectors.toList());
    }
    
    @Override
    public Player getPlayer(UUID id) {
        return onlinePlayers.stream()
                            .filter(player -> player.getUniqueId().equals(id))
                            .findAny()
                            .orElse(null);
    }
    
    @Override
    public BukkitScheduler getScheduler() {
        if (!schedulerStarted) {
            System.err.println("Warning scheduler not started, call TestServer#startScheduler");
        }
        return scheduler;
    }
    
    @Override
    public World createWorld(WorldCreator creator) {
        World world = creator.createWorld();
        worlds.add(world);
        return world;
    }
    
    @Override
    public boolean unloadWorld(String name, boolean save) {
        return unloadWorld(getWorld(name), save);
    }
    
    @Override
    public boolean unloadWorld(World world, boolean save) {
        return worlds.remove(world);
    }
    
    @Override
    public World getWorld(String name) {
        return worlds.stream().filter(world -> world.getName().equals(name)).findAny().orElse(null);
    }
    
    @Override
    public World getWorld(UUID uid) {
        return worlds.stream().filter(world -> world.getUID().equals(uid)).findAny().orElse(null);
    }
    
    @Override
    public Logger getLogger() {
        return Logger.getLogger(getServerName());
    }
    
    @Override
    @Copy(CraftServer.class)
    public abstract PluginCommand getPluginCommand(String name);
    
    @Override
    public Map<String, String[]> getCommandAliases() {
        return Collections.emptyMap();
    }
    
    @Override
    public OfflinePlayer getOfflinePlayer(String name) {
        TestOfflinePlayer op = getInternalGenerator().create(TestOfflinePlayer.class);
        op.setName(name);
        return op;
    }
    
    @Override
    public OfflinePlayer getOfflinePlayer(UUID uuid) {
        TestOfflinePlayer op = getInternalGenerator().create(TestOfflinePlayer.class);
        op.setUniqueId(uuid);
        return op;
    }
    
    @Override
    public void banIP(String address) {
        iPBans.add(address);
    }
    
    @Override
    public void unbanIP(String address) {
        iPBans.remove(address);
    }
    
    @Override
    public Inventory createInventory(InventoryHolder owner, InventoryType type) {
        return new TestInventory(type, owner);
    }
    
    @Override
    public Inventory createInventory(InventoryHolder owner, InventoryType type, String title) {
        return new TestInventory(type, owner, title);
    }
    
    private InventoryType getType(int size) {
        for (InventoryType inventoryType : InventoryType.values()) {
            if (inventoryType.getDefaultSize() == size) {
                return inventoryType;
            }
        }
        return null;
    }
    
    @Override
    public Inventory createInventory(InventoryHolder owner, int size)
            throws IllegalArgumentException {
        return createInventory(owner, getType(size));
    }
    
    @Override
    public Inventory createInventory(InventoryHolder owner, int size, String title)
            throws IllegalArgumentException {
        return createInventory(owner, getType(size), title);
    }
    
    @Override
    public Warning.WarningState getWarningState() {
        return Warning.WarningState.DEFAULT;
    }
    
    @Override
    public void unsubscribeFromDefaultPerms(boolean op, Permissible permissible) {
    }
    
    @Override
    public void unsubscribeFromPermission(String permission, Permissible permissible) {
    }
    
    @Override
    public Set<Permission> getDefaultPermissions(boolean op) {
        return Collections.emptySet();
    }
    
    @Override
    public void subscribeToDefaultPerms(boolean op, Permissible permissible) {
    }
}
