package org.komputing.kethereum.erc1450

import java.math.BigInteger
import kotlin.Boolean
import kotlin.ByteArray
import kotlin.String
import org.kethereum.contract.abi.types.PaginatedByteArray
import org.kethereum.contract.abi.types.encodeTypes
import org.kethereum.contract.abi.types.model.type_params.BitsTypeParams
import org.kethereum.contract.abi.types.model.type_params.BytesTypeParams
import org.kethereum.contract.abi.types.model.types.AddressETHType
import org.kethereum.contract.abi.types.model.types.BoolETHType
import org.kethereum.contract.abi.types.model.types.StringETHType
import org.kethereum.contract.abi.types.model.types.UIntETHType
import org.kethereum.model.Address
import org.kethereum.model.Transaction
import org.kethereum.model.createEmptyTransaction
import org.kethereum.rpc.EthereumRPC

val FourByteName: ByteArray = byteArrayOf(6, -3, -34, 3)

val FourByteTotalSupply: ByteArray = byteArrayOf(24, 22, 13, -35)

val FourByteIssuer: ByteArray = byteArrayOf(29, 20, 56, 72)

val FourByteDecimals: ByteArray = byteArrayOf(49, 60, -27, 103)

val FourByteSymbol: ByteArray = byteArrayOf(-107, -40, -101, 65)

val FourByteIssuanceNumber: ByteArray = byteArrayOf(-26, 94, 49, -85)

val FourByteBalanceOf: ByteArray = byteArrayOf(112, -96, -126, 49)

val FourByteTransferFrom: ByteArray = byteArrayOf(35, -72, 114, -35)

val FourByteMint: ByteArray = byteArrayOf(64, -63, 15, 25)

val FourByteBurnFrom: ByteArray = byteArrayOf(121, -52, 103, -112)

class ERC1450TransactionGenerator(
  address: Address
) {
  private val tx: Transaction = createEmptyTransaction().apply { to = address }

  internal fun nameETHTyped() = tx.copy(input = FourByteName + encodeTypes())
  /**
   * Signature: name()
   * 4Byte: 06fdde03
   */
  fun name(): Transaction = nameETHTyped()

  internal fun totalSupplyETHTyped() = tx.copy(input = FourByteTotalSupply + encodeTypes())
  /**
   * Signature: totalSupply()
   * 4Byte: 18160ddd
   */
  fun totalSupply(): Transaction = totalSupplyETHTyped()

  internal fun issuerETHTyped() = tx.copy(input = FourByteIssuer + encodeTypes())
  /**
   * Signature: issuer()
   * 4Byte: 1d143848
   */
  fun issuer(): Transaction = issuerETHTyped()

  internal fun decimalsETHTyped() = tx.copy(input = FourByteDecimals + encodeTypes())
  /**
   * Signature: decimals()
   * 4Byte: 313ce567
   */
  fun decimals(): Transaction = decimalsETHTyped()

  internal fun symbolETHTyped() = tx.copy(input = FourByteSymbol + encodeTypes())
  /**
   * Signature: symbol()
   * 4Byte: 95d89b41
   */
  fun symbol(): Transaction = symbolETHTyped()

  internal fun issuanceNumberETHTyped() = tx.copy(input = FourByteIssuanceNumber + encodeTypes())
  /**
   * Signature: issuanceNumber()
   * 4Byte: e65e31ab
   */
  fun issuanceNumber(): Transaction = issuanceNumberETHTyped()

  internal fun balanceOfETHTyped(investor: AddressETHType) = tx.copy(input = FourByteBalanceOf +
      encodeTypes(investor))
  /**
   * Signature: balanceOf(address)
   * 4Byte: 70a08231
   */
  fun balanceOf(investor: Address): Transaction =
      balanceOfETHTyped(AddressETHType.ofNativeKotlinType(investor))

  internal fun transferFromETHTyped(
    from: AddressETHType,
    to: AddressETHType,
    value: UIntETHType
  ) = tx.copy(input = FourByteTransferFrom + encodeTypes(from, to, value))
  /**
   * Signature: transferFrom(address,address,uint256)
   * 4Byte: 23b872dd
   */
  fun transferFrom(
    from: Address,
    to: Address,
    value: BigInteger
  ): Transaction = transferFromETHTyped(AddressETHType.ofNativeKotlinType(from),
      AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)))

  internal fun mintETHTyped(to: AddressETHType, value: UIntETHType) = tx.copy(input = FourByteMint +
      encodeTypes(to, value))
  /**
   * Signature: mint(address,uint256)
   * 4Byte: 40c10f19
   */
  fun mint(to: Address, value: BigInteger): Transaction =
      mintETHTyped(AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)))

  internal fun burnFromETHTyped(who: AddressETHType, value: UIntETHType) = tx.copy(input =
      FourByteBurnFrom + encodeTypes(who, value))
  /**
   * Signature: burnFrom(address,uint256)
   * 4Byte: 79cc6790
   */
  fun burnFrom(who: Address, value: BigInteger): Transaction =
      burnFromETHTyped(AddressETHType.ofNativeKotlinType(who),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)))
}

class ERC1450RPCConnector(
  private val address: Address,
  private val rpc: EthereumRPC
) {
  private val txGenerator: ERC1450TransactionGenerator = ERC1450TransactionGenerator(address)

  private fun nameETHTyped(blockSpec: String = "latest"): StringETHType? {
    val tx = txGenerator.nameETHTyped()
    return StringETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: name()
   * 4Byte: 06fdde03
   */
  fun name(blockSpec: String = "latest"): String? = nameETHTyped(blockSpec)?.toKotlinType()

  private fun totalSupplyETHTyped(blockSpec: String = "latest"): UIntETHType? {
    val tx = txGenerator.totalSupplyETHTyped()
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=256))
  }

  /**
   * Signature: totalSupply()
   * 4Byte: 18160ddd
   */
  fun totalSupply(blockSpec: String = "latest"): BigInteger? =
      totalSupplyETHTyped(blockSpec)?.toKotlinType()

  private fun issuerETHTyped(blockSpec: String = "latest"): AddressETHType? {
    val tx = txGenerator.issuerETHTyped()
    return AddressETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: issuer()
   * 4Byte: 1d143848
   */
  fun issuer(blockSpec: String = "latest"): Address? = issuerETHTyped(blockSpec)?.toKotlinType()

  private fun decimalsETHTyped(blockSpec: String = "latest"): UIntETHType? {
    val tx = txGenerator.decimalsETHTyped()
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=8))
  }

  /**
   * Signature: decimals()
   * 4Byte: 313ce567
   */
  fun decimals(blockSpec: String = "latest"): BigInteger? =
      decimalsETHTyped(blockSpec)?.toKotlinType()

  private fun symbolETHTyped(blockSpec: String = "latest"): StringETHType? {
    val tx = txGenerator.symbolETHTyped()
    return StringETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: symbol()
   * 4Byte: 95d89b41
   */
  fun symbol(blockSpec: String = "latest"): String? = symbolETHTyped(blockSpec)?.toKotlinType()

  private fun issuanceNumberETHTyped(blockSpec: String = "latest"): UIntETHType? {
    val tx = txGenerator.issuanceNumberETHTyped()
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=256))
  }

  /**
   * Signature: issuanceNumber()
   * 4Byte: e65e31ab
   */
  fun issuanceNumber(blockSpec: String = "latest"): BigInteger? =
      issuanceNumberETHTyped(blockSpec)?.toKotlinType()

  private fun balanceOfETHTyped(investor: AddressETHType, blockSpec: String = "latest"):
      UIntETHType? {
    val tx = txGenerator.balanceOfETHTyped(investor)
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=256))
  }

  /**
   * Signature: balanceOf(address)
   * 4Byte: 70a08231
   */
  fun balanceOf(investor: Address, blockSpec: String = "latest"): BigInteger? =
      balanceOfETHTyped(AddressETHType.ofNativeKotlinType(investor),blockSpec)?.toKotlinType()

  private fun transferFromETHTyped(
    from: AddressETHType,
    to: AddressETHType,
    value: UIntETHType,
    blockSpec: String = "latest"
  ): BoolETHType? {
    val tx = txGenerator.transferFromETHTyped(from, to, value)
    return BoolETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: transferFrom(address,address,uint256)
   * 4Byte: 23b872dd
   */
  fun transferFrom(
    from: Address,
    to: Address,
    value: BigInteger,
    blockSpec: String = "latest"
  ): Boolean? = transferFromETHTyped(AddressETHType.ofNativeKotlinType(from),
      AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)),blockSpec)?.toKotlinType()

  private fun mintETHTyped(
    to: AddressETHType,
    value: UIntETHType,
    blockSpec: String = "latest"
  ): BoolETHType? {
    val tx = txGenerator.mintETHTyped(to, value)
    return BoolETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: mint(address,uint256)
   * 4Byte: 40c10f19
   */
  fun mint(
    to: Address,
    value: BigInteger,
    blockSpec: String = "latest"
  ): Boolean? = mintETHTyped(AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)),blockSpec)?.toKotlinType()

  private fun burnFromETHTyped(
    who: AddressETHType,
    value: UIntETHType,
    blockSpec: String = "latest"
  ): BoolETHType? {
    val tx = txGenerator.burnFromETHTyped(who, value)
    return BoolETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: burnFrom(address,uint256)
   * 4Byte: 79cc6790
   */
  fun burnFrom(
    who: Address,
    value: BigInteger,
    blockSpec: String = "latest"
  ): Boolean? = burnFromETHTyped(AddressETHType.ofNativeKotlinType(who),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)),blockSpec)?.toKotlinType()
}

class ERC1450TransactionDecoder {
  fun isName(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteName)

  fun isTotalSupply(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteTotalSupply)

  fun isIssuer(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteIssuer)

  fun isDecimals(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteDecimals)

  fun isSymbol(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteSymbol)

  fun isIssuanceNumber(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteIssuanceNumber)

  fun isBalanceOf(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteBalanceOf)

  fun isTransferFrom(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteTransferFrom)

  fun isMint(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteMint)

  fun isBurnFrom(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteBurnFrom)
}
