package net.io_0.maja.validation;

import net.io_0.maja.Property;
import org.apache.commons.validator.routines.*;

import java.math.BigDecimal;
import java.util.Collection;

import static net.io_0.maja.validation.PropertyPredicate.not;

public interface PropertyPredicates {
  PropertyPredicate<?> assigned = Property::isAssigned;

  PropertyPredicate<?> assignedAndNotNull = assigned.and(not(Property::isNull));

  PropertyPredicate<?> unassignedOrNull = not(assignedAndNotNull);

  PropertyPredicate<?> unassignedOrNotNull = not(assigned).or(not(Property::isNull));

  static <T> PropertyPredicate<T> unassignedOrNullOr(PropertyPredicate<T> predicate) {
    return not(Property<T>::isAssigned).or(Property::isNull).or(predicate);
  }

  static PropertyPredicate<Number> lte(Number number) {
    return property -> compare(number, property.getValue()) <= 0;
  }

  static PropertyPredicate<Number> lt(Number number) {
    return property -> compare(number, property.getValue()) < 0;
  }

  static PropertyPredicate<Number> gte(Number number) {
    return property -> compare(number, property.getValue()) >= 0;
  }

  static PropertyPredicate<Number> gt(Number number) {
    return property -> compare(number, property.getValue()) > 0;
  }

  static PropertyPredicate<String> lengthGte(Integer number) {
    return property -> property.getValue().length() >= number;
  }

  static PropertyPredicate<String> lengthLte(Integer number) {
    return property -> property.getValue().length() <= number;
  }

  static PropertyPredicate<Collection<?>> sizeGte(Integer number) {
    return property -> property.getValue().size() >= number;
  }

  static PropertyPredicate<Collection<?>> sizeLte(Integer number) {
    return property -> property.getValue().size() <= number;
  }

  static PropertyPredicate<String> regexMatch(String pattern) {
    return property -> new RegexValidator(pattern).isValid(property.getValue());
  }

  PropertyPredicate<String> email = property -> EmailValidator.getInstance().isValid(property.getValue());

  PropertyPredicate<String> hostname = property -> DomainValidator.getInstance().isValid(property.getValue());

  PropertyPredicate<String> inet4Address = property -> InetAddressValidator.getInstance().isValidInet4Address(property.getValue());

  PropertyPredicate<String> inet6Address = property -> InetAddressValidator.getInstance().isValidInet6Address(property.getValue());

  PropertyPredicate<String> url = property -> UrlValidator.getInstance().isValid(property.getValue());

  static PropertyPredicate<Number> multipleOf(Number number) {
    return property -> new BigDecimal(property.getValue().toString()).remainder(new BigDecimal(number.toString())).abs().floatValue() < 0.0000001;
  }

  static Integer compare(Number a, Number b) {
    return new BigDecimal(a.toString()).compareTo(new BigDecimal(b.toString()));
  }
}
