/*
 * Copyright (C) 2017 XStream Committers.
 * All rights reserved.
 *
 * The software in this package is published under the terms of the BSD
 * style license a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 *
 * Created on 13. January 2017 by Matej Cimbora
 */
package com.feilong.lib.xstream.converters.time;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;

import com.feilong.lib.xstream.converters.ConversionException;
import com.feilong.lib.xstream.converters.SingleValueConverter;

/**
 * Converts a {@link LocalDateTime} to a string.
 *
 * @author Matej Cimbora
 * @since 1.4.10
 */
public class LocalDateTimeConverter implements SingleValueConverter{

    private static final DateTimeFormatter FORMATTER;

    static{
        FORMATTER = new DateTimeFormatterBuilder().appendPattern("uuuu-MM-dd'T'HH:mm:ss")
                        .appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).toFormatter();
    }

    @Override
    public boolean canConvert(@SuppressWarnings("rawtypes") final Class type){
        return LocalDateTime.class == type;
    }

    @Override
    public Object fromString(final String str){
        try{
            return LocalDateTime.parse(str);
        }catch (final DateTimeParseException e){
            final ConversionException exception = new ConversionException("Cannot parse value as local date time", e);
            exception.add("value", str);
            throw exception;
        }
    }

    @Override
    public String toString(final Object obj){
        if (obj == null){
            return null;
        }

        final LocalDateTime localDateTime = (LocalDateTime) obj;
        return FORMATTER.format(localDateTime);
    }
}
