//--------------------------------------------------------------------------
//	Copyright (c) 1998-2004, Drew Davidson and Luke Blanshard
//  All rights reserved.
//
//	Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//
//	Redistributions of source code must retain the above copyright notice,
//  this list of conditions and the following disclaimer.
//	Redistributions in binary form must reproduce the above copyright
//  notice, this list of conditions and the following disclaimer in the
//  documentation and/or other materials provided with the distribution.
//	Neither the name of the Drew Davidson nor the names of its contributors
//  may be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
//  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
//  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
//  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
//  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
//  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
//  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//  AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
//  THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//--------------------------------------------------------------------------
package com.feilong.lib.ognl;

import com.feilong.lib.ognl.enhance.ExpressionAccessor;

/**
 * JJTree interface for AST nodes, as modified to handle the OGNL operations getValue and
 * setValue. JJTree's original comment:
 * 
 * All AST nodes must implement this interface. It provides basic
 * machinery for constructing the parent and child relationships
 * between nodes.
 * 
 * @author Luke Blanshard (blanshlu@netscape.net)
 * @author Drew Davidson (drew@ognl.org)
 */
public interface Node extends JavaSource{

    /**
     * This method is called after the node has been made the current
     * node. It indicates that child nodes can now be added to it.
     */
    public void jjtOpen();

    /**
     * This method is called after all the child nodes have been
     * added.
     */
    public void jjtClose();

    /**
     * This pair of methods are used to inform the node of its
     * parent.
     *
     * @param n
     *            the Node to make the parent of this node.
     */
    public void jjtSetParent(Node n);

    public Node jjtGetParent();

    /**
     * This method tells the node to add its argument to the node's
     * list of children.
     *
     * @param n
     *            the Node to add as a child of this node.
     * @param i
     *            the position at which to add the child node.
     */
    public void jjtAddChild(Node n,int i);

    /**
     * This method returns a child node. The children are numbered
     * from zero, left to right.
     *
     * @param i
     *            the position from which to get the child node.
     * @return the child Node at position i.
     */
    public Node jjtGetChild(int i);

    /**
     * Return the number of children the node has.
     *
     * @return the number of children for this node.
     */
    public int jjtGetNumChildren();

    // OGNL additions to Node:

    /**
     * Extracts the value from the given source object that is appropriate for this node
     * within the given context.
     * 
     * @param context
     *            the OgnlContext within which to perform the operation.
     * @param source
     *            the Object from which to get the value.
     * @return the value from the source (as appropriate within the provided context).
     * @throws OgnlException
     *             if the value get fails.
     */
    public Object getValue(OgnlContext context,Object source) throws OgnlException;

    /**
     * Sets the given value in the given target as appropriate for this node within the
     * given context.
     * 
     * @param context
     *            the OgnlContext within which to perform the operation.
     * @param target
     *            the Object upon which to set the value.
     * @param value
     *            the Object representing the value to apply to the target.
     * @throws OgnlException
     *             if the value set fails.
     */
    public void setValue(OgnlContext context,Object target,Object value) throws OgnlException;

    /**
     * Gets the compiled bytecode enhanced expression accessor for getting/setting values.
     * 
     * @return The accessor for this node, or null if none has been compiled for it.
     */
    ExpressionAccessor getAccessor();

    /**
     * Sets a new compiled accessor for this node expression.
     * 
     * @param accessor
     *            The compiled representation of this node.
     */
    void setAccessor(ExpressionAccessor accessor);
}
