//--------------------------------------------------------------------------
//	Copyright (c) 1998-2004, Drew Davidson and Luke Blanshard
//  All rights reserved.
//
//	Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//
//	Redistributions of source code must retain the above copyright notice,
//  this list of conditions and the following disclaimer.
//	Redistributions in binary form must reproduce the above copyright
//  notice, this list of conditions and the following disclaimer in the
//  documentation and/or other materials provided with the distribution.
//	Neither the name of the Drew Davidson nor the names of its contributors
//  may be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
//  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
//  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
//  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
//  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
//  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
//  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//  AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
//  THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//--------------------------------------------------------------------------
package com.feilong.lib.ognl;

import java.lang.reflect.Member;
import java.util.Map;

/**
 * This interface provides a hook for preparing for accessing members
 * of objects. The Java2 version of this method can allow access
 * to otherwise inaccessable members, such as private fields.
 *
 * @author Luke Blanshard (blanshlu@netscape.net)
 * @author Drew Davidson (drew@ognl.org)
 * @version 15 October 1999
 */
public interface MemberAccess{

    /**
     * Sets the member up for accessibility
     * 
     * @param context
     *            the current execution context.
     * @param target
     *            the Object upon which to perform the setup operation.
     * @param member
     *            the Member upon which to perform the setup operation.
     * @param propertyName
     *            the property upon which to perform the setup operation.
     * @return the Object representing the original accessibility state of the target prior to the setup operation.
     */
    public Object setup(Map context,Object target,Member member,String propertyName);

    /**
     * Restores the member from the previous setup call.
     * 
     * @param context
     *            the current execution context.
     * @param target
     *            the Object upon which to perform the setup operation.
     * @param member
     *            the Member upon which to perform the setup operation.
     * @param propertyName
     *            the property upon which to perform the setup operation.
     * @param state
     *            the Object holding the state to restore (target state prior to the setup operation).
     */
    public void restore(Map context,Object target,Member member,String propertyName,Object state);

    /**
     * Returns true if the given member is accessible or can be made accessible
     * by this object.
     * 
     * @param context
     *            the current execution context.
     * @param target
     *            the Object to test accessibility for.
     * @param member
     *            the Member to test accessibility for.
     * @param propertyName
     *            the property to test accessibility for.
     * @return true if the target/member/propertyName is accessible in the context, false otherwise.
     */
    public boolean isAccessible(Map context,Object target,Member member,String propertyName);
}
