/*
 * Copyright (C) 2008 feilong
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.feilong.lib.excel.ognl;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Member;
import java.lang.reflect.Modifier;
import java.util.Map;

import com.feilong.lib.ognl.MemberAccess;

/**
 * This class provides methods for setting up and restoring
 * access in a Field. Java 2 provides access utilities for setting
 * and getting fields that are non-public. This object provides
 * coarse-grained access controls to allow access to private, protected
 * and package protected members. This will apply to all classes
 * and members.
 *
 * @author Luke Blanshard (blanshlu@netscape.net)
 * @author Drew Davidson (drew@ognl.org)
 * @version 15 October 1999
 */
public class DefaultMemberAccess implements MemberAccess{

    /** The allow private access. */
    private boolean allowPrivateAccess          = false;

    /** The allow protected access. */
    private boolean allowProtectedAccess        = false;

    /** The allow package protected access. */
    private boolean allowPackageProtectedAccess = false;

    //---------------------------------------------------------------

    public DefaultMemberAccess(boolean allowAllAccess){
        this(allowAllAccess, allowAllAccess, allowAllAccess);
    }

    public DefaultMemberAccess(boolean allowPrivateAccess, boolean allowProtectedAccess, boolean allowPackageProtectedAccess){
        super();
        this.allowPrivateAccess = allowPrivateAccess;
        this.allowProtectedAccess = allowProtectedAccess;
        this.allowPackageProtectedAccess = allowPackageProtectedAccess;
    }

    //---------------------------------------------------------------

    @Override
    public Object setup(Map context,Object target,Member member,String propertyName){
        Object result = null;

        if (isAccessible(context, target, member, propertyName)){
            AccessibleObject accessible = (AccessibleObject) member;

            if (!accessible.isAccessible()){
                result = Boolean.FALSE;

                accessible.setAccessible(true);
            }
        }
        return result;
    }

    @Override
    public void restore(Map context,Object target,Member member,String propertyName,Object state){
        if (state == null){
            return;
        }

        boolean stateboolean = ((Boolean) state).booleanValue(); // Using twice (avoid unboxing)
        if (!stateboolean){
            AccessibleObject accessible = (AccessibleObject) member;
            accessible.setAccessible(stateboolean);
            return;
        }

        throw new IllegalArgumentException(
                        "Improper restore state [" + stateboolean + "] for target [" + target + "], member [" + member + "], propertyName ["
                                        + propertyName + "]");
    }

    /**
     * Returns true if the given member is accessible or can be made accessible
     * by this object.
     *
     * @param context
     *            the current execution context (not used).
     * @param target
     *            the Object to test accessibility for (not used).
     * @param member
     *            the Member to test accessibility for.
     * @param propertyName
     *            the property to test accessibility for (not used).
     * @return true if the member is accessible in the context, false otherwise.
     */
    @Override
    public boolean isAccessible(Map context,Object target,Member member,String propertyName){
        int modifiers = member.getModifiers();
        boolean result = Modifier.isPublic(modifiers);
        if (result){
            return true;
        }

        //---------------------------------------------------------------
        if (Modifier.isPrivate(modifiers)){
            return allowPrivateAccess;
        }
        if (Modifier.isProtected(modifiers)){
            return allowProtectedAccess;
        }
        return allowPackageProtectedAccess;
    }

}