/*
 * Copyright (C) 2008 feilong
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.feilong.excel;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import com.feilong.lib.excel.convertor.BigDecimalConvertor;
import com.feilong.lib.excel.convertor.BooleanConvertor;
import com.feilong.lib.excel.convertor.DataConvertor;
import com.feilong.lib.excel.convertor.DateConvertor;
import com.feilong.lib.excel.convertor.DoubleConvertor;
import com.feilong.lib.excel.convertor.IntegerConvertor;
import com.feilong.lib.excel.convertor.LongConvertor;
import com.feilong.lib.excel.convertor.StringConvertor;

/**
 * The Class DataConvertorConfigurator.
 */
public class DataConvertorConfig implements Serializable{

    /** The Constant serialVersionUID. */
    private static final long                     serialVersionUID = -6172555487692156540L;

    //---------------------------------------------------------------

    /** The supports map. */
    private final Map<String, Class<?>>           supportsMap      = new HashMap<>();

    /** The convertor map. */
    private final Map<Class<?>, DataConvertor<?>> convertorMap     = new HashMap<>();

    /** The instance. */
    private static DataConvertorConfig            instance;

    //---------------------------------------------------------------

    /**
     * Instantiates a new data convertor configurator.
     */
    private DataConvertorConfig(){
        registerDataConvertor(new StringConvertor());
        registerDataConvertor(new IntegerConvertor());
        registerDataConvertor(new LongConvertor());
        registerDataConvertor(new DoubleConvertor());
        registerDataConvertor(new BigDecimalConvertor());
        registerDataConvertor(new DateConvertor());
        registerDataConvertor(new BooleanConvertor());
    }

    //---------------------------------------------------------------
    /**
     * Register data convertor.
     *
     * @param dataConvertor
     *            the dc
     */
    public void registerDataConvertor(DataConvertor<?> dataConvertor){
        supportsMap.put(dataConvertor.getDataTypeAbbr(), dataConvertor.supportClass());
        convertorMap.put(dataConvertor.supportClass(), dataConvertor);
    }

    //---------------------------------------------------------------

    /**
     * Gets the convertor.
     *
     * @param <T>
     *            the generic type
     * @param clazz
     *            the clazz
     * @return the convertor
     */
    @SuppressWarnings("unchecked")
    public <T> DataConvertor<T> getConvertor(Class<T> clazz){
        return (DataConvertor<T>) convertorMap.get(clazz);
    }

    /**
     * Gets the supported class.
     *
     * @param name
     *            the name
     * @return the supported class
     */
    public Class<?> getSupportedClass(String name){
        return supportsMap.get(name);
    }

    //---------------------------------------------------------------

    /**
     * Gets the single instance of DataConvertorConfigurator.
     *
     * @return single instance of DataConvertorConfigurator
     */
    public static DataConvertorConfig getInstance(){
        if (instance == null){
            instance = new DataConvertorConfig();
        }
        return instance;
    }
}
