/*
 * Decompiled with CFR 0.152.
 */
package smile.regression;

import java.util.Arrays;
import java.util.Properties;
import smile.math.BFGS;
import smile.math.DifferentiableMultivariateFunction;
import smile.math.MathEx;
import smile.math.kernel.MercerKernel;
import smile.math.matrix.Matrix;
import smile.regression.Regression;
import smile.stat.distribution.MultivariateGaussianDistribution;

public class GaussianProcessRegression<T>
implements Regression<T> {
    private static final long serialVersionUID = 2L;
    public final MercerKernel<T> kernel;
    public final T[] regressors;
    public final double[] w;
    public final double mean;
    public final double sd;
    public final double noise;
    public final double L;
    private final Matrix.Cholesky cholesky;

    public GaussianProcessRegression(MercerKernel<T> kernel, T[] regressors, double[] weight, double noise) {
        this(kernel, regressors, weight, noise, 0.0, 1.0);
    }

    public GaussianProcessRegression(MercerKernel<T> kernel, T[] regressors, double[] weight, double noise, double mean, double sd) {
        this(kernel, regressors, weight, noise, mean, sd, null, Double.NaN);
    }

    public GaussianProcessRegression(MercerKernel<T> kernel, T[] regressors, double[] weight, double noise, double mean, double sd, Matrix.Cholesky cholesky, double L) {
        if (noise < 0.0) {
            throw new IllegalArgumentException("Invalid noise variance: " + noise);
        }
        this.kernel = kernel;
        this.regressors = regressors;
        this.w = weight;
        this.noise = noise;
        this.mean = mean;
        this.sd = sd;
        this.cholesky = cholesky;
        this.L = L;
    }

    @Override
    public double predict(T x) {
        int n = this.regressors.length;
        double mu = 0.0;
        for (int i = 0; i < n; ++i) {
            mu += this.w[i] * this.kernel.k(x, this.regressors[i]);
        }
        return mu * this.sd + this.mean;
    }

    public double predict(T x, double[] estimation) {
        if (this.cholesky == null) {
            throw new UnsupportedOperationException("The Cholesky decomposition of kernel matrix is not available.");
        }
        int n = this.regressors.length;
        double[] k = new double[n];
        for (int i = 0; i < n; ++i) {
            k[i] = this.kernel.k(x, this.regressors[i]);
        }
        double[] Kx = this.cholesky.solve(k);
        double mu = MathEx.dot((double[])this.w, (double[])k);
        double sd = Math.sqrt(this.kernel.k(x, x) - MathEx.dot((double[])Kx, (double[])k));
        mu = mu * this.sd + this.mean;
        estimation[0] = mu;
        estimation[1] = sd *= this.sd;
        return mu;
    }

    public JointPrediction query(T[] samples) {
        if (this.cholesky == null) {
            throw new UnsupportedOperationException("The Cholesky decomposition of kernel matrix is not available.");
        }
        Matrix Kx = this.kernel.K((Object[])samples);
        Matrix Kt = this.kernel.K((Object[])samples, (Object[])this.regressors);
        Matrix Kv = Kt.transpose(false);
        this.cholesky.solve(Kv);
        Matrix cov = Kx.sub(Kt.mm(Kv));
        cov.mul(this.sd * this.sd);
        double[] mu = Kt.mv(this.w);
        double[] std = cov.diag();
        int m = samples.length;
        for (int i = 0; i < m; ++i) {
            mu[i] = mu[i] * this.sd + this.mean;
            std[i] = Math.sqrt(std[i]);
        }
        return new JointPrediction(samples, mu, std, cov);
    }

    public String toString() {
        StringBuffer sb = new StringBuffer("GaussianProcessRegression {\n");
        sb.append("  kernel: ").append(this.kernel).append(",\n");
        sb.append("  regressors: ").append(this.regressors.length).append(",\n");
        sb.append("  mean: ").append(String.format("%.4f,\n", this.mean));
        sb.append("  std.dev: ").append(String.format("%.4f,\n", this.sd));
        sb.append("  noise: ").append(String.format("%.4f", this.noise));
        if (!Double.isNaN(this.L)) {
            sb.append(",\n  log marginal likelihood: ").append(String.format("%.4f", this.L));
        }
        sb.append("\n}");
        return sb.toString();
    }

    public static GaussianProcessRegression<double[]> fit(double[][] x, double[] y, Properties params) {
        MercerKernel kernel = MercerKernel.of((String)params.getProperty("smile.gaussian_process.kernel", "linear"));
        double noise = Double.parseDouble(params.getProperty("smile.gaussian_process.noise", "1E-10"));
        boolean normalize = Boolean.parseBoolean(params.getProperty("smile.gaussian_process.normalize", "true"));
        double tol = Double.parseDouble(params.getProperty("smile.gaussian_process.tolerance", "1E-5"));
        int maxIter = Integer.parseInt(params.getProperty("smile.gaussian_process.iterations", "0"));
        return GaussianProcessRegression.fit(x, y, kernel, noise, normalize, tol, maxIter);
    }

    public static <T> GaussianProcessRegression<T> fit(T[] x, double[] y, MercerKernel<T> kernel, Properties params) {
        double noise = Double.parseDouble(params.getProperty("smile.gaussian_process.noise", "1E-10"));
        boolean normalize = Boolean.parseBoolean(params.getProperty("smile.gaussian_process.normalize", "true"));
        double tol = Double.parseDouble(params.getProperty("smile.gaussian_process.tolerance", "1E-5"));
        int maxIter = Integer.parseInt(params.getProperty("smile.gaussian_process.iterations", "0"));
        return GaussianProcessRegression.fit(x, y, kernel, noise, normalize, tol, maxIter);
    }

    public static <T> GaussianProcessRegression<T> fit(T[] x, double[] y, MercerKernel<T> kernel, double noise) {
        return GaussianProcessRegression.fit(x, y, kernel, noise, true, 1.0E-5, 0);
    }

    public static <T> GaussianProcessRegression<T> fit(T[] x, double[] y, MercerKernel<T> kernel, double noise, boolean normalize, double tol, int maxIter) {
        if (x.length != y.length) {
            throw new IllegalArgumentException(String.format("The sizes of X and Y don't match: %d != %d", x.length, y.length));
        }
        if (noise < 0.0) {
            throw new IllegalArgumentException("Invalid noise variance = " + noise);
        }
        int n = x.length;
        double mean = 0.0;
        double sd = 1.0;
        if (normalize) {
            mean = MathEx.mean((double[])y);
            sd = MathEx.sd((double[])y);
            double[] target = new double[n];
            for (int i = 0; i < n; ++i) {
                target[i] = (y[i] - mean) / sd;
            }
            y = target;
        }
        if (maxIter > 0) {
            LogMarginalLikelihood<T> objective = new LogMarginalLikelihood<T>(x, y, kernel);
            double[] hp = kernel.hyperparameters();
            double[] lo = kernel.lo();
            double[] hi = kernel.hi();
            int m = lo.length;
            double[] params = Arrays.copyOf(hp, m + 1);
            double[] l = Arrays.copyOf(lo, m + 1);
            double[] u = Arrays.copyOf(hi, m + 1);
            params[m] = noise;
            l[m] = 1.0E-10;
            u[m] = 100000.0;
            BFGS.minimize(objective, (int)5, (double[])params, (double[])l, (double[])u, (double)tol, (int)maxIter);
            kernel = kernel.of(params);
            noise = params[params.length - 1];
        }
        Matrix K = kernel.K((Object[])x);
        K.addDiag(noise);
        Matrix.Cholesky cholesky = K.cholesky(true);
        double[] w = cholesky.solve(y);
        double L = -0.5 * (MathEx.dot((double[])y, (double[])w) + cholesky.logdet() + (double)n * Math.log(Math.PI * 2));
        return new GaussianProcessRegression<T>(kernel, x, w, noise, mean, sd, cholesky, L);
    }

    public static <T> GaussianProcessRegression<T> fit(T[] x, double[] y, T[] t, MercerKernel<T> kernel, Properties params) {
        double noise = Double.parseDouble(params.getProperty("smile.gaussian_process.noise", "1E-10"));
        boolean normalize = Boolean.parseBoolean(params.getProperty("smile.gaussian_process.normalize", "true"));
        return GaussianProcessRegression.fit(x, y, t, kernel, noise, normalize);
    }

    public static <T> GaussianProcessRegression<T> fit(T[] x, double[] y, T[] t, MercerKernel<T> kernel, double noise) {
        return GaussianProcessRegression.fit(x, y, t, kernel, noise, true);
    }

    public static <T> GaussianProcessRegression<T> fit(T[] x, double[] y, T[] t, MercerKernel<T> kernel, double noise, boolean normalize) {
        if (x.length != y.length) {
            throw new IllegalArgumentException(String.format("The sizes of X and Y don't match: %d != %d", x.length, y.length));
        }
        if (noise < 0.0) {
            throw new IllegalArgumentException("Invalid noise variance = " + noise);
        }
        double mean = 0.0;
        double sd = 1.0;
        if (normalize) {
            mean = MathEx.mean((double[])y);
            sd = MathEx.sd((double[])y);
            int n = x.length;
            double[] target = new double[n];
            for (int i = 0; i < n; ++i) {
                target[i] = (y[i] - mean) / sd;
            }
            y = target;
        }
        Matrix G = kernel.K((Object[])x, (Object[])t);
        Matrix K = G.ata();
        Matrix Kt = kernel.K((Object[])t);
        K.add(noise, Kt);
        Matrix.LU lu = K.lu(true);
        double[] Gty = G.tv(y);
        double[] w = lu.solve(Gty);
        return new GaussianProcessRegression<T>(kernel, t, w, noise, mean, sd);
    }

    public static <T> GaussianProcessRegression<T> nystrom(T[] x, double[] y, T[] t, MercerKernel<T> kernel, Properties params) {
        double noise = Double.parseDouble(params.getProperty("smile.gaussian_process.noise", "1E-10"));
        boolean normalize = Boolean.parseBoolean(params.getProperty("smile.gaussian_process.normalize", "true"));
        return GaussianProcessRegression.nystrom(x, y, t, kernel, noise, normalize);
    }

    public static <T> GaussianProcessRegression<T> nystrom(T[] x, double[] y, T[] t, MercerKernel<T> kernel, double noise) {
        return GaussianProcessRegression.nystrom(x, y, t, kernel, noise, true);
    }

    public static <T> GaussianProcessRegression<T> nystrom(T[] x, double[] y, T[] t, MercerKernel<T> kernel, double noise, boolean normalize) {
        if (x.length != y.length) {
            throw new IllegalArgumentException(String.format("The sizes of X and Y don't match: %d != %d", x.length, y.length));
        }
        if (noise < 0.0) {
            throw new IllegalArgumentException("Invalid noise variance = " + noise);
        }
        int n = x.length;
        int m = t.length;
        double mean = 0.0;
        double sd = 1.0;
        if (normalize) {
            mean = MathEx.mean((double[])y);
            sd = MathEx.sd((double[])y);
            double[] target = new double[n];
            for (int i = 0; i < n; ++i) {
                target[i] = (y[i] - mean) / sd;
            }
            y = target;
        }
        Matrix E = kernel.K((Object[])x, (Object[])t);
        Matrix W = kernel.K((Object[])t);
        Matrix.EVD eigen = W.eigen(false, true, true).sort();
        Matrix U = eigen.Vr;
        Matrix D = eigen.diag();
        for (int i = 0; i < m; ++i) {
            D.set(i, i, 1.0 / Math.sqrt(D.get(i, i)));
        }
        Matrix UD = U.mm(D);
        Matrix UDUt = UD.mt(U);
        Matrix L = E.mm(UDUt);
        Matrix LtL = L.ata();
        LtL.addDiag(noise);
        Matrix.Cholesky chol = LtL.cholesky(true);
        Matrix invLtL = chol.inverse();
        Matrix Kinv = L.mm(invLtL).mt(L);
        double[] w = Kinv.tv(y);
        for (int i = 0; i < n; ++i) {
            w[i] = (y[i] - w[i]) / noise;
        }
        return new GaussianProcessRegression<T>(kernel, x, w, noise, mean, sd);
    }

    public class JointPrediction {
        public final T[] x;
        public final double[] mu;
        public final double[] sd;
        public final Matrix cov;
        private MultivariateGaussianDistribution dist;

        public JointPrediction(T[] x, double[] mu, double[] sd, Matrix cov) {
            this.x = x;
            this.mu = mu;
            this.sd = sd;
            this.cov = cov;
        }

        public double[][] sample(int n) {
            if (this.dist == null) {
                this.dist = new MultivariateGaussianDistribution(this.mu, this.cov);
            }
            return this.dist.rand(n);
        }

        public String toString() {
            return String.format("GaussianProcessRegression.Prediction {\n  mean    = %s\n  std.dev = %s\n  cov     = %s\n}", Arrays.toString(this.mu), Arrays.toString(this.sd), this.cov.toString(true));
        }
    }

    private static class LogMarginalLikelihood<T>
    implements DifferentiableMultivariateFunction {
        final T[] x;
        final double[] y;
        MercerKernel<T> kernel;

        public LogMarginalLikelihood(T[] x, double[] y, MercerKernel<T> kernel) {
            this.x = x;
            this.y = y;
            this.kernel = kernel;
        }

        public double f(double[] params) {
            this.kernel = this.kernel.of(params);
            double noise = params[params.length - 1];
            Matrix K = this.kernel.K((Object[])this.x);
            K.addDiag(noise);
            Matrix.Cholesky cholesky = K.cholesky(true);
            double[] w = cholesky.solve(this.y);
            int n = this.x.length;
            double L = -0.5 * (MathEx.dot((double[])this.y, (double[])w) + cholesky.logdet() + (double)n * Math.log(Math.PI * 2));
            return -L;
        }

        public double g(double[] params, double[] g) {
            this.kernel = this.kernel.of(params);
            double noise = params[params.length - 1];
            Matrix[] K = this.kernel.KG((Object[])this.x);
            Matrix Ky = K[0];
            Ky.addDiag(noise);
            Matrix.Cholesky cholesky = Ky.cholesky(true);
            Matrix Kinv = cholesky.inverse();
            double[] w = Kinv.mv(this.y);
            g[g.length - 1] = -(MathEx.dot((double[])w, (double[])w) - Kinv.trace()) / 2.0;
            for (int i = 1; i < g.length; ++i) {
                Matrix Kg = K[i];
                double gi = Kg.xAx(w) - Kinv.mm(Kg).trace();
                g[i - 1] = -gi / 2.0;
            }
            int n = this.x.length;
            double L = -0.5 * (MathEx.dot((double[])this.y, (double[])w) + cholesky.logdet() + (double)n * Math.log(Math.PI * 2));
            return -L;
        }
    }
}

