/*
 * Copyright 2017 Fs2 Rabbit
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.github.gvolpe.fs2rabbit

import cats.data.{IndexedStateT, State}
import cats.effect.{IO, Sync}
import com.github.gvolpe.fs2rabbit.model.QueueName
import com.github.gvolpe.fs2rabbit.xalgebra._
import com.github.gvolpe.fs2rabbit.xstate._
import fs2.Stream

object Experiment extends App {

  val connectionImpl  = new ConnectionImpl[IO]
  val declarationImpl = new DeclarationImpl[IO]
  val bindingImpl     = new BindingImpl[IO]
  val consumingImpl   = new ConsumingImpl[IO]

  val daQ = new QueueName("daQ")

  val program =
    for {
      _ <- connectionImpl.connect
      d <- declarationImpl.declareQueue(daQ)
      b <- bindingImpl.bindQueue(daQ)
      s <- consumingImpl.consume(daQ)
    } yield s"$d > $b > $s"

  program.runA(Disconnected).compile.last.unsafeRunSync().foreach(println)

//  program.run(Disconnected).run.unsafeRunSync()

  val a: State[AMQPState, Stream[IO, String]] = State(_ => (Connected, Stream.eval(IO("s"))))

  a.runA(Disconnected).value

}

class ConnectionImpl[F[_]](implicit F: Sync[F]) extends AMQPConnection[ConnectionT[Stream[F, ?], ?]] {
  override def connect: ConnectionT[Stream[F, ?], Unit] = {
    IndexedStateT { _ =>
      Stream.eval(F.delay((Connected, println("connection established"))))
    }
  }
}

class DeclarationImpl[F[_]](implicit F: Sync[F]) extends Declaration[QueueDeclarationT[Stream[F, ?], ?]] {
  override def declareQueue(queueName: QueueName): QueueDeclarationT[Stream[F, ?], String] = {
    IndexedStateT { _ =>
      Stream.eval(F.delay((QueueDeclared, s"declared ${queueName.value}")))
    }
  }
}

class BindingImpl[F[_]](implicit F: Sync[F]) extends Binding[QueueBindingT[Stream[F, ?], ?]] {
  override def bindQueue(queueName: QueueName): QueueBindingT[Stream[F, ?], String] = {
    IndexedStateT { _ =>
      Stream.eval(F.delay((QueueBound, s"bound ${queueName.value}")))
    }
  }
}

class ConsumingImpl[F[_]](implicit F: Sync[F]) extends Consuming[ConsumingT[Stream[F, ?], ?]] {
  override def consume(queueName: QueueName): ConsumingT[Stream[F, ?], String] = {
    IndexedStateT { _ =>
      Stream.eval(F.delay((Consuming, s"consuming from ${queueName.value}")))
    }
  }
}

object xstate {
  sealed trait AMQPState
  case object Disconnected extends AMQPState
  case object Connected extends AMQPState
  case object QueueDeclared extends AMQPState
  case object QueueBound extends AMQPState
  case object Consuming extends AMQPState

  type ConnectionT[F[_], A] = IndexedStateT[F, Disconnected.type, Connected.type, A]
  type QueueDeclarationT[F[_], A] = IndexedStateT[F, Connected.type, QueueDeclared.type, A]
  type QueueBindingT[F[_], A] = IndexedStateT[F, QueueDeclared.type, QueueBound.type, A]
  type ConsumingT[F[_], A] = IndexedStateT[F, QueueBound.type, Consuming.type, A]
}

object xalgebra {

  trait AMQPConnection[F[_]] {
    def connect: F[Unit]
  }

  trait Binding[F[_]] {
    def bindQueue(queueName: QueueName): F[String]
  }

  trait Declaration[F[_]] {
    def declareQueue(queueName: QueueName): F[String]
  }

  trait Consuming[F[_]] {
    def consume(queueName: QueueName): F[String]
  }

}
