package com.github.fengyuchenglun.core.visitor.springmvc;

import com.github.fengyuchenglun.core.Context;
import com.github.fengyuchenglun.core.Options;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Set;

/**
 * Spring @Controller 解析工具
 */
public class Controllers {

    /**
     * The constant CONTROLLER.
     */
    public static final String CONTROLLER = "Controller";
    /**
     * The constant REST_CONTROLLER.
     */
    public static final String REST_CONTROLLER = "RestController";
    /**
     * The constant RESPONSE_BODY.
     */
    public static final String RESPONSE_BODY = "ResponseBody";

    /**
     * The constant ANNOTATIONS.
     */
    public List<String> annotations = Lists.newArrayList();
    public Set<String> responseBodys = Sets.newHashSet();

    /**
     * Instantiates a new Controllers.
     */
    private Controllers() {
        Options options = Context.getApigcc().getOptions();
        annotations.addAll(options.getControllerSets());
        responseBodys=options.getResponseBodySets();
    }

    /**
     * 类级的内部类，也就是静态的成员式内部类，该内部类的实例与外部类的实例
     * 没有绑定关系，而且只有被调用到才会装载，从而实现了延迟加载
     */
    private static class SingletonHolder {
        /**
         * 静态初始化器，由JVM来保证线程安全
         */
        private static Controllers instance = new Controllers();
    }

    /**
     * Gets instance.
     *
     * @return the instance
     */
    public static Controllers getInstance() {
        return SingletonHolder.instance;
    }


    /**
     * Accept boolean.
     *
     * @param nodes the nodes
     * @return the boolean
     */
    public boolean accept(NodeList<AnnotationExpr> nodes) {
        for (int i = 0; i < nodes.size(); i++) {
            if (accept(nodes.get(i))) {
                return true;
            }
        }
        return false;
    }

    /**
     * Accept boolean.
     *
     * @param n the n
     * @return the boolean
     */
    public boolean accept(AnnotationExpr n) {
        if (!annotations.contains(n.getNameAsString())) {
            return false;
        }
        return true;
    }

    /**
     * Is response body boolean.
     *
     * @param n the n
     * @return the boolean
     */
    public boolean isResponseBody(ClassOrInterfaceDeclaration n) {
        return responseBodys.stream().anyMatch(x -> {
            return n.isAnnotationPresent(x);
        });
    }
}
