package com.github.fengyuchenglun.core;

import com.github.fengyuchenglun.core.common.loging.Logger;
import com.github.fengyuchenglun.core.common.loging.LoggerFactory;
import com.github.fengyuchenglun.core.visitor.Framework;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.Getter;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * The type Options.
 */
@Getter
public class Options {

    /**
     * The Log.
     */
    private Logger log = LoggerFactory.getLogger(this.getClass());

    /**
     * The constant DEFAULT_PRODUCTION.
     */
    public static final Path DEFAULT_PRODUCTION = Paths.get(Context.NAME);
    /**
     * The constant DEFAULT_SOURCE_STRUCTURE.
     */
    public static final Path DEFAULT_SOURCE_STRUCTURE = Paths.get("src", "main", "java");
    /**
     * The constant DEFAULT_PROJECT_PATH.
     */
    public static final Path DEFAULT_PROJECT_PATH = Paths.get(System.getProperty("user.dir"));
    /**
     * The constant DEFAULT_OUT.
     */
    public static final Path DEFAULT_OUT = Paths.get("build");


    /**
     * The constant CONTROLLER.
     */
    public static final String CONTROLLER = "Controller";
    /**
     * The constant REST_CONTROLLER.
     */
    public static final String REST_CONTROLLER = "RestController";
    /**
     * The constant RESPONSE_BODY.
     */
    public static final String RESPONSE_BODY = "ResponseBody";

    /**
     * project根目录
     */
    private Path project = DEFAULT_PROJECT_PATH;

    /**
     * 项目名称 生成 index.json index.adoc index.html
     */
    private String id = "index";

    /**
     * 文档标题
     */
    private String title;

    /**
     * 文档描述
     */
    private String description;
    /**
     * 文档版本
     */
    private String version;

    /**
     * source code folder wait for parse
     * or just some code
     * default: parse user.dir 's code
     */
    private Set<Path> sources = Sets.newHashSet();

    /**
     * dependency source code folders
     */
    private Set<Path> dependencies = Sets.newHashSet();

    /**
     * dependency third jars
     */
    private Set<Path> jars = Sets.newHashSet();

    /**
     * 输出文件包裹的文件夹
     */
    private Path production = DEFAULT_PRODUCTION;

    /**
     * 输出目录
     */
    private Path out = DEFAULT_OUT;

    /**
     * The Css.
     */
    private String css;

    /**
     * The Ignores.
     */
    private Set<String> ignores = Sets.newHashSet();

    /**
     * The Is show valid field.
     */
    private Boolean isShowValidField = false;

    /**
     * 保存需要扫描的controller注解名称
     */
    private Set<String> controllerSets= Sets.newHashSet(Arrays.asList(CONTROLLER,REST_CONTROLLER));
    /**
     * 用于确认是否返回的json请求
     */
    private Set<String> responseBodySets = Sets.newHashSet(RESPONSE_BODY,REST_CONTROLLER);

    /**
     * Project options.
     *
     * @param value the value
     * @return the options
     */
    public Options project(Path value) {
        this.project = value;
        return this;
    }

    /**
     * Id options.
     *
     * @param value the value
     * @return the options
     */
    public Options id(String value) {
        this.id = value;
        return this;
    }

    /**
     * Title options.
     *
     * @param value the value
     * @return the options
     */
    public Options title(String value) {
        this.title = value;
        return this;
    }

    /**
     * Description options.
     *
     * @param value the value
     * @return the options
     */
    public Options description(String value) {
        this.description = value;
        return this;
    }

    /**
     * Version options.
     *
     * @param value the value
     * @return the options
     */
    public Options version(String value) {
        this.version = value;
        return this;
    }

    /**
     * Source options.
     *
     * @param values the values
     * @return the options
     */
    public Options source(Path... values) {
        for (Path value : values) {
            if (!value.isAbsolute()) {
                value = project.resolve(value);
            }
            if (Files.exists(value)) {
                this.sources.add(value);
            }
        }
        dependency(values);
        return this;
    }

    /**
     * Dependency options.
     *
     * @param values the values
     * @return the options
     */
    public Options dependency(Path... values) {
        for (Path value : values) {
            if (!value.isAbsolute()) {
                value = project.resolve(value);
            }
            if (Files.exists(value)) {
                this.dependencies.add(value);
            }
        }
        return this;
    }

    /**
     * Jar options.
     *
     * @param values the values
     * @return the options
     */
    public Options jar(Path... values) {
        for (Path value : values) {
            if (!value.isAbsolute()) {
                value = project.resolve(value);
            }
            if (!Files.exists(value)) {
                continue;
            }
            if (!Files.isDirectory(value) && value.toString().endsWith("jar")) {
                this.jars.add(value);
            }
            try {
                Files.list(value).forEach(this::jar);
            } catch (IOException e) {
                log.debug("read list of {} error", value);
            }
        }
        return this;
    }

    /**
     * Production options.
     *
     * @param value the value
     * @return the options
     */
    public Options production(Path value) {
        this.production = value;
        return this;
    }

    /**
     * Is show valid field options.
     *
     * @param value the value
     * @return the options
     */
    public Options isShowValidField(Boolean value) {
        this.isShowValidField = value;
        return this;
    }

    /**
     * Out options.
     *
     * @param value the value
     * @return the options
     */
    public Options out(Path value) {
        this.out = value;
        return this;
    }

    /**
     * Css options.
     *
     * @param css the css
     * @return the options
     */
    public Options css(String css) {
        this.css = css;
        return this;
    }

    /**
     * Ignore options.
     *
     * @param values the values
     * @return the options
     */
    public Options ignore(String... values) {
        ignores = Sets.newHashSet(values);
        return this;
    }

    /**
     * Framework options.
     *
     * @param framework the framework
     * @return the options
     */
    public Options framework(Framework framework) {
        Framework.current = framework;
        return this;
    }

    /**
     * Gets out path.
     *
     * @return the out path
     */
    public Path getOutPath() {
        if (out.isAbsolute()) {
            return out.resolve(production);
        }
        return project.resolve(out).resolve(production);
    }

    /**
     * Controller set options.
     *
     * @param controllerList the controller list
     * @return the options
     */
    public Options controllerSet(List<String> controllerList) {
        this.controllerSets.addAll(controllerList);
        return this;
    }

    /**
     * Response body set options.
     *
     * @param responseBodyList the response body list
     * @return the options
     */
    public Options responseBodySet(List<String> responseBodyList) {
        this.responseBodySets.addAll(responseBodyList);
        return this;
    }
}
