package com.github.fengyuchenglun.core;

import com.github.fengyuchenglun.core.common.loging.Logger;
import com.github.fengyuchenglun.core.common.loging.LoggerFactory;
import com.github.fengyuchenglun.core.handler.TreeHandler;
import com.github.fengyuchenglun.core.schema.Bucket;
import com.github.fengyuchenglun.core.visitor.Framework;
import com.github.javaparser.ParseResult;
import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.symbolsolver.JavaSymbolSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.JarTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.JavaParserTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ReflectionTypeSolver;
import com.github.javaparser.utils.SourceRoot;
import com.google.common.collect.Lists;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

/**
 * 🐷 工具入口类、上下文
 */
public class Apigcc extends Context {

    private Logger log = LoggerFactory.getLogger(this.getClass());

    public Apigcc() {
        this(new Options());
    }

    public Apigcc(String root) {
        this(new Options().source(Paths.get(root)));
    }

    public Apigcc(Options options) {
        super();
        this.options = options;
        this.tree.setId(options.getId());
        this.tree.setName(options.getTitle());
        this.tree.setDescription(options.getDescription());
        this.tree.setVersion(options.getVersion());
        this.tree.setBucket(new Bucket(options.getId()));
        // 设置忽略类型
        this.getIgnoreTypes().addAll(options.getIgnores());
    }

    /**
     * 解析源代码
     * @return
     */
    private List<CompilationUnit> parseSource(){

        List<CompilationUnit> cus = Lists.newLinkedList();
//        解析配置
        ParserConfiguration parserConfiguration = buildParserConfiguration();
        for (Path path : options.getSources()) {
            SourceRoot root = new SourceRoot(path, parserConfiguration);
            try {
                for (ParseResult<CompilationUnit> result : root.tryToParse()) {
                    if(result.isSuccessful() && result.getResult().isPresent()){
                        cus.add(result.getResult().get());
                    }
                }
            } catch (IOException e) {
                log.warning("parse source error : {}", root.getRoot());
            }
        }

        return cus;
    }

    /**
     * 搜寻给定代码及依赖环境
     * 找到Endpoints，构建Tree
     *
     * @return
     */
    public Apigcc lookup() {
        // 解析源码之后的值
        List<CompilationUnit> cus = parseSource();

        //  获得当前框架环境
        Framework framework = Framework.getCurrent(cus);

        for (CompilationUnit cu : cus) {
            cu.accept(framework.getVisitor(),this.tree);
        }

        Integer totalNodes = tree.getBucket().getGroups().stream()
                .map(g -> g.getNodes().size())
                .reduce(0, (sum, i) -> sum += i);
        log.info("\r\nFound {} Controllers, {} Endpoints", tree.getBucket().getGroups().size(), totalNodes);

        return this;
    }

    /**
     * 执行默认的构建任务
     */
    public void build() {
        getPipeline().forEach(this::build);
    }

    public void build(TreeHandler... handlers) {
        Arrays.stream(handlers).forEach(this::build);
    }

    public void build(TreeHandler handler) {
        handler.handle(tree, options);
    }


    /**
     * 构建代码解析所需的环境
     * @return
     */
    private ParserConfiguration buildParserConfiguration() {
        // 设置默认值
        if (options.getSources().isEmpty()) {
            options.source(options.getProject().resolve(Options.DEFAULT_SOURCE_STRUCTURE));
        }

        getTypeSolver().add(new ReflectionTypeSolver());
        // 依赖包
        options.getDependencies().forEach(value -> getTypeSolver().add(new JavaParserTypeSolver(value)));
        options.getJars().forEach(value -> {
            try {
                getTypeSolver().add(new JarTypeSolver(value));
            } catch (IOException e) {
                log.debug("read jar fail:{}",value);
            }
        });
        //  设置解析配置
        ParserConfiguration parserConfiguration = new ParserConfiguration();
        parserConfiguration.setSymbolResolver(new JavaSymbolSolver(getTypeSolver()));
        return parserConfiguration;
    }

}
