/*
 * Copyright 2002-2013 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jn.agileway.web.security.csrf;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.jn.langx.util.Preconditions;
import com.jn.langx.util.logging.Loggers;
import org.slf4j.Logger;


/**
 * {@link CsrfAuthenticationStrategy} is in charge of removing the {@link CsrfToken} upon
 * authenticating. A new {@link CsrfToken} will then be generated by the framework upon
 * the next request.
 */
public final class CsrfAuthenticationStrategy {

    private final Logger logger = Loggers.getLogger(getClass());

    private final CsrfTokenRepository csrfTokenRepository;

    /**
     * Creates a new instance
     *
     * @param csrfTokenRepository the {@link CsrfTokenRepository} to use
     */
    public CsrfAuthenticationStrategy(CsrfTokenRepository csrfTokenRepository) {
        Preconditions.checkNotNull(csrfTokenRepository, "csrfTokenRepository cannot be null" );
        this.csrfTokenRepository = csrfTokenRepository;
    }

    public void onAuthentication(HttpServletRequest request, HttpServletResponse response) {
        boolean containsToken = this.csrfTokenRepository.loadToken(request) != null;
        if (containsToken) {
            this.csrfTokenRepository.saveToken(null, request, response);
            CsrfToken newToken = this.csrfTokenRepository.generateToken(request);
            this.csrfTokenRepository.saveToken(newToken, request, response);
            request.setAttribute(CsrfToken.class.getName(), newToken);
            request.setAttribute(newToken.getParameterName(), newToken);
            this.logger.debug("Replaced CSRF Token" );
        }
    }

}
