/*
 * Decompiled with CFR 0.152.
 */
package oshi.util;

import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ParseUtil {
    private static final Logger LOG = LoggerFactory.getLogger(ParseUtil.class);
    private static final String DEFAULT_LOG_MSG = "{} didn't parse. Returning default. {}";
    private static final Pattern HERTZ_PATTERN = Pattern.compile("(\\d+(.\\d+)?) ?([kMGT]?Hz).*");
    private static final Pattern VALID_HEX = Pattern.compile("[0-9a-fA-F]+");
    private static final DateTimeFormatter CIM_FORMAT = DateTimeFormatter.ofPattern("yyyyMMddHHmmss.SSSSSSZZZZZ", Locale.US);
    private static final Pattern DHMS = Pattern.compile("(?:(\\d+)-)?(?:(\\d+):)?(\\d+):(\\d+)(?:\\.(\\d+))?");
    private static final Pattern UUID_PATTERN = Pattern.compile(".*([0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}).*");
    private static final String HZ = "Hz";
    private static final String KHZ = "kHz";
    private static final String MHZ = "MHz";
    private static final String GHZ = "GHz";
    private static final String THZ = "THz";
    private static final String PHZ = "PHz";
    private static final Map<String, Long> multipliers = new HashMap<String, Long>();

    private ParseUtil() {
    }

    public static long parseHertz(String hertz) {
        Matcher matcher = HERTZ_PATTERN.matcher(hertz.trim());
        if (matcher.find() && matcher.groupCount() == 3) {
            Double value = Double.valueOf(matcher.group(1)) * (double)multipliers.getOrDefault(matcher.group(3), -1L).longValue();
            return value < 0.0 ? -1L : value.longValue();
        }
        return -1L;
    }

    public static int parseLastInt(String s, int i) {
        try {
            return Integer.parseInt(ParseUtil.parseLastString(s));
        }
        catch (NumberFormatException e) {
            LOG.trace(DEFAULT_LOG_MSG, (Object)s, (Object)e);
            return i;
        }
    }

    public static String parseLastString(String s) {
        String[] ss = s.split("\\s+");
        if (ss.length < 1) {
            return s;
        }
        return ss[ss.length - 1];
    }

    public static byte[] hexStringToByteArray(String digits) {
        int len = digits.length();
        if (!VALID_HEX.matcher(digits).matches() || (len & 1) != 0) {
            LOG.warn("Invalid hexadecimal string: {}", (Object)digits);
            return new byte[0];
        }
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte)(Character.digit(digits.charAt(i), 16) << 4 | Character.digit(digits.charAt(i + 1), 16));
        }
        return data;
    }

    public static byte[] stringToByteArray(String text, int length) {
        return Arrays.copyOf(text.getBytes(), length);
    }

    public static byte[] longToByteArray(long value, int valueSize, int length) {
        long val = value;
        byte[] b = new byte[8];
        for (int i = 7; i >= 0 && val != 0L; val >>>= 8, --i) {
            b[i] = (byte)val;
        }
        return Arrays.copyOfRange(b, 8 - valueSize, 8 + length - valueSize);
    }

    public static long strToLong(String str, int size) {
        return ParseUtil.byteArrayToLong(str.getBytes(), size);
    }

    public static long byteArrayToLong(byte[] bytes, int size) {
        if (size > 8) {
            throw new IllegalArgumentException("Can't convert more than 8 bytes.");
        }
        if (size > bytes.length) {
            throw new IllegalArgumentException("Size can't be larger than array length.");
        }
        long total = 0L;
        for (int i = 0; i < size; ++i) {
            total = total << 8 | (long)(bytes[i] & 0xFF);
        }
        return total;
    }

    public static float byteArrayToFloat(byte[] bytes, int size, int fpBits) {
        return (float)ParseUtil.byteArrayToLong(bytes, size) / (float)(1 << fpBits);
    }

    public static long cimDateTimeToMillis(String cimDate) {
        try {
            int tzInMinutes = Integer.parseInt(cimDate.substring(22));
            LocalTime offsetAsLocalTime = LocalTime.MIN.plusMinutes(tzInMinutes);
            OffsetDateTime dateTime = OffsetDateTime.parse(cimDate.substring(0, 22) + offsetAsLocalTime.format(DateTimeFormatter.ISO_LOCAL_TIME), CIM_FORMAT);
            return dateTime.toInstant().toEpochMilli();
        }
        catch (IndexOutOfBoundsException | NumberFormatException | DateTimeParseException e) {
            LOG.trace(DEFAULT_LOG_MSG, (Object)cimDate, (Object)e);
            return 0L;
        }
    }

    public static String hexStringToString(String hexString) {
        if (hexString.length() % 2 > 0) {
            return hexString;
        }
        StringBuilder sb = new StringBuilder();
        try {
            for (int pos = 0; pos < hexString.length(); pos += 2) {
                int charAsInt = Integer.parseInt(hexString.substring(pos, pos + 2), 16);
                if (charAsInt < 32 || charAsInt > 127) {
                    return hexString;
                }
                sb.append((char)charAsInt);
            }
        }
        catch (NumberFormatException e) {
            LOG.trace(DEFAULT_LOG_MSG, (Object)hexString, (Object)e);
            return hexString;
        }
        return sb.toString();
    }

    public static int parseIntOrDefault(String s, int defaultInt) {
        try {
            return Integer.parseInt(s);
        }
        catch (NumberFormatException e) {
            LOG.trace(DEFAULT_LOG_MSG, (Object)s, (Object)e);
            return defaultInt;
        }
    }

    public static long parseLongOrDefault(String s, long defaultLong) {
        try {
            return Long.parseLong(s);
        }
        catch (NumberFormatException e) {
            LOG.trace(DEFAULT_LOG_MSG, (Object)s, (Object)e);
            return defaultLong;
        }
    }

    public static double parseDoubleOrDefault(String s, double defaultDouble) {
        try {
            return Double.parseDouble(s);
        }
        catch (NumberFormatException e) {
            LOG.trace(DEFAULT_LOG_MSG, (Object)s, (Object)e);
            return defaultDouble;
        }
    }

    public static long parseDHMSOrDefault(String s, long defaultLong) {
        Matcher m = DHMS.matcher(s);
        if (m.matches()) {
            long milliseconds = 0L;
            if (m.group(1) != null) {
                milliseconds += ParseUtil.parseLongOrDefault(m.group(1), 0L) * 86400000L;
            }
            if (m.group(2) != null) {
                milliseconds += ParseUtil.parseLongOrDefault(m.group(2), 0L) * 3600000L;
            }
            milliseconds += ParseUtil.parseLongOrDefault(m.group(3), 0L) * 60000L;
            milliseconds += ParseUtil.parseLongOrDefault(m.group(4), 0L) * 1000L;
            milliseconds = (long)((double)milliseconds + 1000.0 * ParseUtil.parseDoubleOrDefault("0." + m.group(5), 0.0));
            return milliseconds;
        }
        return defaultLong;
    }

    public static String parseUuidOrDefault(String s, String defaultStr) {
        Matcher m = UUID_PATTERN.matcher(s.toLowerCase());
        if (m.matches()) {
            return m.group(1);
        }
        return defaultStr;
    }

    public static String getSingleQuoteStringValue(String line) {
        String[] split = line.split("'");
        if (split.length < 2) {
            return "";
        }
        return split[1];
    }

    public static int getFirstIntValue(String line) {
        String[] split = line.split("=");
        if (split.length < 2) {
            return 0;
        }
        return ParseUtil.parseIntOrDefault(split[1].trim().split("\\s+")[0], 0);
    }

    static {
        multipliers.put(HZ, 1L);
        multipliers.put(KHZ, 1000L);
        multipliers.put(MHZ, 1000000L);
        multipliers.put(GHZ, 1000000000L);
        multipliers.put(THZ, 1000000000000L);
        multipliers.put(PHZ, 1000000000000000L);
    }
}

