/*
 * [The "BSD licence"]
 * Copyright (c) 2013-2015 Dandelion
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Dandelion nor the names of its contributors 
 * may be used to endorse or promote products derived from this software 
 * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.github.dandelion.datatables.core.extension.feature;

import com.github.dandelion.core.util.StringUtils;

/**
 * <p>
 * POJO used to wrap an extra HTML snippet that end-users can add around the
 * table.
 * <p>
 * Each HTML snippet will result in a dynamic feature that will added before the
 * DataTables initialization. The feature can be then activated thanks to the
 * <code>dom</code>/<code>dt:dom</code> attributes (JSP/Thymeleaf).
 * 
 * @author Thibault Duchateau
 * @since 0.10.0
 */
public class ExtraHtml {

   private String uid;
   private String container;
   private String cssStyle;
   private String cssClass;
   private String content;

   public ExtraHtml() {
      this.container = "div";
   }

   public String getUid() {
      return uid;
   }

   public void setUid(String uid) {
      this.uid = uid;
   }

   public String getContainer() {
      return container;
   }

   public void setContainer(String container) {
      this.container = container;
   }

   public String getCssStyle() {
      return cssStyle;
   }

   public void setCssStyle(String cssStyle) {
      this.cssStyle = cssStyle;
   }

   public String getCssClass() {
      return cssClass;
   }

   public void setCssClass(String cssClass) {
      this.cssClass = cssClass;
   }

   public String getContent() {
      return content;
   }

   public void setContent(String content) {
      this.content = content;
   }

   public StringBuilder getJavascript() {

      StringBuilder js = new StringBuilder();

      js.append("$.fn.dataTable.ext.feature.push({");
      js.append("\"fnInit\": function( settings ){");
      js.append("var container = document.createElement('" + this.container + "');");

      if (StringUtils.isNotBlank(this.cssClass)) {
         js.append("$(container).attr('class', '" + this.cssClass + "');");
      }
      if (StringUtils.isNotBlank(this.cssStyle)) {
         js.append("$(container).attr('style', '" + this.cssStyle + "');");
      }

      js.append("$(container).html('" + this.content.replaceAll("'", "&quot;") + "');");
      js.append("return container;");
      js.append("},");
      js.append("\"cFeature\": \"" + this.uid + "\",");
      js.append("\"sFeature\": \"" + "Group" + this.uid + "\"");
      js.append("} );");

      return js;
   }

   @Override
   public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((container == null) ? 0 : container.hashCode());
      result = prime * result + ((content == null) ? 0 : content.hashCode());
      result = prime * result + ((cssClass == null) ? 0 : cssClass.hashCode());
      result = prime * result + ((cssStyle == null) ? 0 : cssStyle.hashCode());
      result = prime * result + ((uid == null) ? 0 : uid.hashCode());
      return result;
   }

   @Override
   public boolean equals(Object obj) {
      if (this == obj)
         return true;
      if (obj == null)
         return false;
      if (getClass() != obj.getClass())
         return false;
      ExtraHtml other = (ExtraHtml) obj;
      if (container == null) {
         if (other.container != null)
            return false;
      }
      else if (!container.equals(other.container))
         return false;
      if (content == null) {
         if (other.content != null)
            return false;
      }
      else if (!content.equals(other.content))
         return false;
      if (cssClass == null) {
         if (other.cssClass != null)
            return false;
      }
      else if (!cssClass.equals(other.cssClass))
         return false;
      if (cssStyle == null) {
         if (other.cssStyle != null)
            return false;
      }
      else if (!cssStyle.equals(other.cssStyle))
         return false;
      if (uid == null) {
         if (other.uid != null)
            return false;
      }
      else if (!uid.equals(other.uid))
         return false;
      return true;
   }

   @Override
   public String toString() {
      return "ExtraHtml [uid=" + uid + ", container=" + container + ", cssStyle=" + cssStyle + ", cssClass=" + cssClass
            + ", content=" + content + "]";
   }
}
