package com.crabshue.commons.validations.file.validators;

import java.io.File;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import com.crabshue.commons.validations.file.annotations.AccessMode;
import com.crabshue.commons.validations.file.annotations.ValidDirectory;

public class DirectoryValidator implements ConstraintValidator<ValidDirectory, File> {

    private AccessMode[] accessModes;

    @Override
    public void initialize(ValidDirectory constraintAnnotation) {
        accessModes = constraintAnnotation.accessModes();
    }

    @Override
    public boolean isValid(File file, ConstraintValidatorContext constraintValidatorContext) {

        if (file != null) {
            if (!file.exists()) {
                setErrorMessage(constraintValidatorContext, "The directory '${validatedValue}' does not exist");
                return false;
            }

            if (!file.isDirectory()) {
                setErrorMessage(constraintValidatorContext, "'${validatedValue}' is not a directory");
                return false;
            }

            for (AccessMode accessMode : accessModes) {
                if (!accessMode.canAccessInMode(file)) {
                    setErrorMessage(constraintValidatorContext, "The directory '${validatedValue}' cannot be accessed in " + accessMode.getAccessMode());
                    return false;
                }
            }
        }

        return true;
    }

    private void setErrorMessage(ConstraintValidatorContext constraintValidatorContext, String message) {
        constraintValidatorContext.disableDefaultConstraintViolation();

        constraintValidatorContext
            .buildConstraintViolationWithTemplate(message)
            .addConstraintViolation();
    }
}
