package com.crabshue.commons.file.validation;

import java.io.File;

import com.crabshue.commons.exceptions.ValidationException;
import com.crabshue.commons.file.exceptions.FileErrorContext;
import com.crabshue.commons.file.exceptions.FileErrorType;
import lombok.NonNull;

/**
 * Utility class for common validations on file and folders.
 *
 */
public class FileValidationUtils {

    /**
     * Check that a {@link File folder} is actually a folder.
     *
     * @param folder the folder.
     */
    public static void validateFolder(@NonNull final File folder) {

        if (!folder.isDirectory()) {
            throw new ValidationException(FileErrorType.NOT_A_FOLDER)
                .addContextValue(FileErrorContext.FOLDER, folder);
        }
    }


    /**
     * Check that a {@link File file} is actually a file.
     *
     * @param file file
     */
    public static void validateFile(@NonNull final File file) {

        if (!file.isFile()) {
            throw new ValidationException(FileErrorType.NOT_A_FILE)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }

    /**
     * Check that a {@link File file/folder} is writable.
     *
     * @param file the file/folder.
     */
    public static void validateWritable(@NonNull final File file) {

        if (!file.canWrite()) {
            throw new ValidationException(FileErrorType.NOT_WRITABLE)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }

    /**
     * Check that a {@link File file/folder} is readable.
     *
     * @param file the file/folder.
     */
    public static void validateReadable(@NonNull final File file) {

        if (!file.canRead()) {
            throw new ValidationException(FileErrorType.NOT_READABLE)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }

    /**
     * Check that a {@link File file/folder} is executable.
     *
     * @param file the file/folder.
     */
    public static void validateExecutable(@NonNull final File file) {

        if (!file.canWrite()) {
            throw new ValidationException(FileErrorType.NOT_EXECUTABLE)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }
}
