package com.crabshue.commons.file;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;

import com.crabshue.commons.exceptions.ApplicationException;
import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.file.exceptions.FileErrorContext;
import com.crabshue.commons.file.exceptions.FileErrorType;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * Utility class for operations on the file system.
 *
 */
@Slf4j
public class FileSystemUtils {

    /**
     * Return or create a folder.
     *
     * @param folderPath the path to the folder
     * @return the already existing / created folder
     */
    public static File provideFolder(@NonNull final String folderPath) {

        return provideFolder(new File(folderPath));
    }

    /**
     * Return or create a folder.
     *
     * @param folder the folder
     * @return the already existing / created folder
     */
    public static File provideFolder(@NonNull final File folder) {

        if (!folder.exists()) {
            try {
                FileUtils.forceMkdir(folder);
                logger.info("Folder created [{}]", folder);
            } catch (IOException e) {
                throw new SystemException(FileErrorType.ERROR_CREATING_FOLDER, e)
                    .addContextValue(FileErrorContext.FOLDER, folder);
            }
        }
        return folder;
    }


    /**
     * Return or create a subfolder inside the given parent folder.
     *
     * @param subfolder the name of the subfolder
     * @param parent    the parent folder
     * @return the already existing / created subfolder
     */
    public static File provideSubFolder(@NonNull final String subfolder,
                                        @NonNull final File parent) {

        File ret = new File(parent, subfolder);

        return provideFolder(ret);
    }

    /**
     * Retrieve a file by name in a folder.
     *
     * @param name   the name of the file
     * @param folder the folder
     * @return the requested file
     */
    public static File retrieveFileInFolder(@NonNull final String name,
                                            @NonNull final File folder) {

        File ret = new File(folder, name);

        if (!ret.exists() || !ret.isFile()) {
            throw new ApplicationException(FileErrorType.NO_FILE_FOUND, "No file found in the folder with given name")
                .addContextValue(FileErrorContext.FOLDER, folder)
                .addContextValue(FileErrorContext.FILENAME, name);
        }
        logger.info("File [{}] found in folder [{}]", ret, folder);
        return ret;
    }

    /**
     * Copy a source file to a target file.
     *
     * @param file       the source file.
     * @param outputFile the target file.
     * @return the target file.
     */
    public static File copyFile(@NonNull final File file,
                                @NonNull final File outputFile) {

        try {
            FileUtils.copyFile(file, outputFile);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_COPY_FILE, e)
                .addContextValue(FileErrorContext.FILE, file)
                .addContextValue(FileErrorContext.FILE, outputFile);
        }

        return outputFile;
    }

    /**
     * Copy a file to a target folder.
     *
     * @param file         the file
     * @param targetFolder the target folder
     * @return the copied file
     */
    public static File copyFileToFolder(@NonNull final File file,
                                        @NonNull final File targetFolder) {

        if (!targetFolder.isDirectory()) {
            throw new IllegalArgumentException("target is not a directory " + targetFolder);
        }

        if (!file.isFile()) {
            throw new IllegalArgumentException("Given file is not a file " + file);
        }

        File target = new File(targetFolder, file.getName());
        try {
            FileUtils.copyFile(file, target);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_COPY_FILE, "Cannot copy file", e)
                .addContextValue(FileErrorContext.FILE, file)
                .addContextValue(FileErrorContext.FILE, target);
        }
        logger.info("Copied file [{}] to [{}]", file, target);
        return target;
    }

    /**
     * Copy a folder to a target folder.
     *
     * @param sourceFolder the source folder
     * @param targetFolder the target folder
     * @return the copied folder
     */
    public static File copyFolder(@NonNull final File sourceFolder,
                                  @NonNull final File targetFolder) {

        if (!sourceFolder.isDirectory()) {
            throw new IllegalArgumentException("at least one of the parameters is not a directory");
        }

        try {
            FileUtils.copyDirectory(sourceFolder, targetFolder);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_COPY_FILE, "Cannot copy folder", e)
                .addContextValue(FileErrorContext.FOLDER, sourceFolder)
                .addContextValue(FileErrorContext.FOLDER, targetFolder);
        }
        logger.info("Copied folder [{}] to [{}]", sourceFolder, targetFolder);

        return targetFolder;
    }

    /**
     * Check if the content of a folder is empty (i.e. doesn't contain any folder nor file)
     *
     * @param folder the folder
     * @return true if the folder is empty, false otherwise
     */
    public static boolean isFolderEmpty(@NonNull final File folder) {

        Collection<File> files = FileUtils.listFilesAndDirs(folder, FileFilterUtils.trueFileFilter(), FileFilterUtils.trueFileFilter());
        // files contains the parent directory
        boolean ret = files.size() <= 1;

        logger.info("Folder [{}] is empty ? : {}", folder, ret);
        return ret;
    }

    /**
     * Retrieve or create a file on the file system, given a folder and a filename.
     *
     * @param folder   the folder.
     * @param filename the filename.
     * @return the file.
     */
    public static File retrieveOrCreateFile(@NonNull final File folder,
                                            @NonNull final String filename) {

        final File ret = new File(folder, filename);

        return retrieveOrCreateFile(ret);
    }

    /**
     * Retrieve or create a file.
     *
     * @param file the file
     * @return the file
     */
    public static File retrieveOrCreateFile(@NonNull final File file) {

        if (!file.exists()) {
            FileSystemUtils.provideFolder(file.getParentFile());
            try {
                FileUtils.touch(file);
                logger.info("Created file [{}]", file);
            } catch (IOException e) {
                throw new SystemException(FileErrorType.ERROR_WRITING_FILE, e)
                    .addContextValue(FileErrorContext.FILE, file);
            }
        } else {
            logger.info("Retrieved file [{}]", file);
        }

        return file;
    }

    /**
     * Check if a file is empty.
     *
     * @param file the file
     * @return {@code true} if the file is empty; {@code false} otherwise.
     */
    public static boolean isFileEmpty(@NonNull final File file) {
        try {
            final List<String> lines = FileUtils.readLines(file, StandardCharsets.UTF_8);
            return lines.isEmpty();
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_READING_FILE, "Cannot read file", e)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }

    /**
     * Clean a folder from its content (file and sub-folders).
     *
     * @param folder the folder.
     */
    public static void cleanDirectory(@NonNull final File folder) {

        if (!folder.exists()) {
            logger.info("Folder [{}] does not exist. No need to clean.", folder);
            return;
        }

        try {
            FileUtils.cleanDirectory(folder);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_CLEANING_FOLDER, e)
                .addContextValue(FileErrorContext.FOLDER, folder);
        }
    }

}
