package com.crabshue.commons.exceptions.utils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Utility class providing methods to format list and maps of values that belong in an exception context.
 *
 */
public class ExceptionMessageUtils {

    private static final String VALUE_SEPARATOR = ", ";

    protected ExceptionMessageUtils() {
    }

    /**
     * Print a {@link Collection} as a {@link String}.
     * <p>Loop through the collection and apply a function to print the elements of the collection.</p>
     *
     * @param collection         the collection.
     * @param printValueFunction the print function.
     * @param <T>                the type of the elements in the collection.
     * @return the formatted print.
     */
    public static <T> String printList(final Collection<T> collection, final Function<T, String> printValueFunction) {
        return collection.stream()
            .filter(Objects::nonNull)
            .map(printValueFunction)
            .collect(Collectors.joining(VALUE_SEPARATOR));
    }

    /**
     * Print a {@link Collection} as a {@link String}.
     * <p>Loop through the collection and apply a function to print the elements of the collection.</p>
     *
     * @param collection         the collection.
     * @param printValueFunction the print function.
     * @param <T>                the type of the elements in the collection.
     * @return the formatted print.
     */
    public static <T> String printValues(final Function<T, String> printValueFunction, final T... collection) {
        return Arrays.stream(collection)
            .filter(Objects::nonNull)
            .map(printValueFunction)
            .collect(Collectors.joining(VALUE_SEPARATOR));
    }

    /**
     * Print a {@link Map} as a {@link String}.
     * <p>Loop through the map and apply a function to print the elements (key, value_) of the map.</p>
     *
     * @param map        the collection.
     * @param printKey   the print function for the key.
     * @param printValue the print function for the value.
     * @param <T>        the type of the elements for the keys.
     * @param <V>        the type of the elements for the values.
     * @return the formatted print.
     */
    public static <T, V> String printMap(final Map<T, V> map, final Function<T, String> printKey, final Function<V, String> printValue) {
        return map.keySet().stream()
            .filter(Objects::nonNull)
            .map((k) ->
                new StringBuilder()
                    .append("(")
                    .append(printKey.apply(k))
                    .append(":")
                    .append(Objects.isNull(map.get(k)) ? "" : printValue.apply(map.get(k)))
                    .append(")")
                    .toString()
            )
            .collect(Collectors.joining(VALUE_SEPARATOR));
    }

}
