/**
 * Chartbeat Android API by Mike Dai Wang.
 * (c) Chartbeat 2016
 */
package com.chartbeat.androidsdk;

import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.net.Uri;

import java.util.Collection;
import java.util.concurrent.TimeUnit;

import rx.Observable;
import rx.Subscriber;
import rx.Subscription;
import rx.schedulers.Schedulers;

/**
 * This class is the main entry point into the Chartbeat SDK. All Chartbeat
 * android SDK functionality is accessible through this class's public static
 * methods.
 * 
 * Generally speaking, all applications <strong>must</strong> call the following
 * methods as part of their normal operation:
 * <ul>
 * <li><strong>startTrackerWithAccountId():</strong> call one of these methods
 * once at application launch. If this is called multiple times, only the first
 * call will have any effect -- subsequent calls will be ignored.
 * <li><strong>trackView():</strong> call one of these methods every time the view changes. This
 * can be done by calling this function in the onResume() function of your
 * activity. This not only updates the view, if necessary, but also ensures that
 * the tracker knows when the app is in the foreground and engaged.
 * <li><strong>userLeftView():</strong> call this when the user leaves the view. This
 * can be done by calling the function in the onPause() function of your activity.
 * <li><strong>userInteracted():</strong> call this every time the user
 * interacts with the application and the application, such as touching the
 * screen. This can be done by calling the function in onUserInteracted()
 * function of your activity.
 * </ul>
 * 
 * All other methods are optional, and allow you to, for example, track scroll position,
 * authors, when the user types and so on.
 * 
 * 
 * @author bjorn
 * @author Mike Dai Wang
 */
public final class Tracker {
    private static final String TAG = "ChartBeat Tracker";

    public static boolean DEBUG_MODE = false;

    // Amount of time to wait until we assume the user has left the app
    private static final int BACKGROUND_IDLE_WAIT_LIMIT_MS = 4000;
    private static Subscription appIdleSubscription;

    private static Context appContext;
    private static String accountID;
    private static String domain;

    static final String KEY_SDK_ACTION_TYPE = "KEY_SDK_ACTION_TYPE";

    static final String ACTION_INIT_TRACKER = "ACTION_INIT_TRACKER";
    static final String ACTION_SET_APP_REFERRER = "ACTION_SET_APP_REFERRER";
    static final String ACTION_STOP_TRACKER = "ACTION_STOP_TRACKER";
    static final String ACTION_TRACK_VIEW = "ACTION_TRACK_VIEW";
    static final String ACTION_LEFT_VIEW = "ACTION_LEFT_VIEW";
    static final String ACTION_USER_INTERACTED = "ACTION_USER_INTERACTED";
    static final String ACTION_USER_TYPED = "ACTION_USER_TYPED";
    static final String ACTION_SET_DOMAIN = "ACTION_SET_DOMAIN";
    static final String ACTION_SET_SUBDOMAIN = "ACTION_SET_SUBDOMAIN";
    static final String ACTION_SET_ZONES = "ACTION_SET_ZONES";
    static final String ACTION_SET_AUTHORS = "ACTION_SET_AUTHORS";
    static final String ACTION_SET_SECTIONS = "ACTION_SET_SECTIONS";
    static final String ACTION_SET_SUBSCRIPTION_STATE = "ACTION_SET_SUBSCRIPTION_STATE";
    static final String ACTION_SET_VIEW_LOADING_TIME = "ACTION_SET_VIEW_LOADING_TIME";
    static final String ACTION_SET_POSITION = "ACTION_SET_POSITION";
    static final String ACTION_PAUSE_TRACKER = "ACTION_PAUSE_TRACKER";
    static final String ACTION_RESTART_PING_SERVICE = "ACTION_RESTART_PING_SERVICE";
    static final String ACTION_BACKGROUND_TRACKER = "ACTION_BACKGROUND_TRACKER";

    static final String KEY_ACCOUNT_ID = "KEY_ACCOUNT_ID";
    static final String KEY_DOMAIN = "KEY_DOMAIN";
    static final String KEY_APP_REFERRER = "KEY_APP_REFERRER";
    static final String KEY_VIEW_ID = "KEY_VIEW_ID";
    static final String KEY_VIEW_TITLE = "KEY_VIEW_TITLE";
    static final String KEY_SUBDOMAIN = "KEY_SUBDOMAIN";
    static final String KEY_ZONES = "KEY_ZONES";
    static final String KEY_SUBSCRIPTION_STATE = "KEY_SUBSCRIPTION_STATE";
    static final String KEY_AUTHORS = "KEY_AUTHORS";
    static final String KEY_SECTIONS = "KEY_SECTIONS";
    static final String KEY_VIEW_LOADING_TIME = "KEY_VIEW_LOADING_TIME";

    static final String KEY_POSITION_TOP = "KEY_POSITION_TOP";
    static final String KEY_WINDOW_HEIGHT = "KEY_WINDOW_HEIGHT";
    static final String KEY_CONTENT_HEIGHT = "KEY_CONTENT_HEIGHT";
    static final String KEY_DOC_WIDTH = "KEY_DOC_WIDTH";

    private static Subscription userInteractSubscription;
    private static final int USER_INTERACT_WINDOW_IN_MILLISECONDS = 500;

    /** ----------- Public static functions -------------- */

    /**
     * initializes the tracker. If the tracker has already been initialized,
     * this call will be ignored.
     *
     * @param accountId
     *            your account id on the Chartbeat system. e.g. "12345"
     * @param domain
     *            the chartbeat dashboard domain name you want to report analytics data to, e.g.
     *            "mynewspaper.com"
     * @param context
     *            the context.
     */
    public static void setupTracker(String accountId, String domain, Context context) {
        if (accountId == null) {
            throw new NullPointerException("accountId cannot be null");
        }

        if (context == null) {
            throw new NullPointerException("context cannot be null");
        }

        if (!(context instanceof Application)) {
            throw new IllegalArgumentException("Application level context is required to initialize Chartbeat Android SDK");
        }
        AwsLogger.initInstance(context,accountId,domain);
        startSDK(accountId, domain, context);
    }

    private static void startSDK(String accountID, String domain, Context context) {
        try {
            appContext = context.getApplicationContext();
            monitorAppStatus();
            Tracker.accountID = accountID;
            Tracker.domain = domain;
            Intent intent = new Intent(context.getApplicationContext(), ChartbeatService.class);
            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_INIT_TRACKER);
            intent.putExtra(KEY_ACCOUNT_ID, accountID);
            if (domain != null) {
                intent.putExtra(KEY_DOMAIN, domain);
            }

            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    public static void restartPingService(Context context) {
        try {
            Intent intent = new Intent(context.getApplicationContext(), ChartbeatService.class);
            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_RESTART_PING_SERVICE);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    private static void monitorAppStatus() {
        try {
            ForegroundTracker.init((Application) appContext);

            ForegroundTracker.get(appContext).addListener(new ForegroundTracker.Listener() {
                @Override
                public void onForegrounded() {
                    restartPingService(appContext);
                    if (appIdleSubscription != null && !appIdleSubscription.isUnsubscribed()) {
                        appIdleSubscription.unsubscribe();
                    }
                }

                @Override
                public void onBackgrounded() {
                    if (appIdleSubscription == null || appIdleSubscription.isUnsubscribed()) {
                        appIdleSubscription = Observable.timer(BACKGROUND_IDLE_WAIT_LIMIT_MS, TimeUnit.MILLISECONDS)
                                .subscribe(new Subscriber<Long>() {
                                    @Override
                                    public void onCompleted() {

                                    }

                                    @Override
                                    public void onError(Throwable e) {

                                    }

                                    @Override
                                    public void onNext(Long aLong) {
                                        backgroundTracker();
                                    }
                                });
                    }
                }
            });
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this to set the app referrer. This is a referrer that is external to
     * the app, such as another app or website. This should be called
     * immediately before calling trackView. If the tracker has not been
     * initialized, this call will be ignored.
     *
     * @param appReferrer
     *            the string representing the appReferrer.
     */
    public static void setAppReferrer(String appReferrer) {
        didInit();
        Intent intent = new Intent(appContext, ChartbeatService.class);
        intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_APP_REFERRER);
        intent.putExtra(KEY_APP_REFERRER, appReferrer);
        sendServiceSignal(intent);
    }

    /**
     * Call this to set the app's push referrer. This is the id of the push alert referrer
     * This should be called
     * immediately before calling trackView. If the tracker has not been
     * initialized, this call will be ignored.
     *
     * @param pushReferrer
     *            the string representing the pushReferrer.
     */
    public static void setPushReferrer(String pushReferrer) {
        pushReferrer = Uri.encode(pushReferrer);
        pushReferrer = "push/?id=" + pushReferrer;
        setAppReferrer(pushReferrer);
    }

    /**
     * Stops the tracker if one has been previously started. Most apps will not
     * need to call this function.
     */
    public static void stopTracker() {
        if (appContext == null) {
            return;
        }

        Intent intent = new Intent(appContext, ChartbeatService.class);
        
        intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_STOP_TRACKER);

        sendServiceSignal(intent);
    }

    /**
     * Pauses the tracker for idle applications
     */
    public static void pauseTracker() {
        if (appContext == null) {
            return;
        }

        Intent intent = new Intent(appContext, ChartbeatService.class);

        intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_PAUSE_TRACKER);

        sendServiceSignal(intent);
    }

    /**
     * Pauses the tracker when app goes to Background
     */
    public static void backgroundTracker() {
        if (appContext == null) {
            return;
        }

        Intent intent = new Intent(appContext, ChartbeatService.class);

        intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_BACKGROUND_TRACKER);

        sendServiceSignal(intent);
    }

    /**
     * Call this whenever you display a new view. Use this in views where you are not tracking position.
     *  If the tracker has not been initialized, this call will be ignored.
     *
     * @param context
     *            the context of the current view being tracked
     * 
     * @param viewId
     *            the id of the view being displayed. Must not be null.
     * @param viewTitle
     *            the title of the view. may be null.
     */
    public static void trackView(Context context, String viewId, String viewTitle) {
        try {
            didInit();

            if (context == null) {
                throw new NullPointerException("context cannot be null");
            }

            if (viewId == null) {
                throw new NullPointerException("viewId cannot be null");
            }

            resetUserInteractionMonitor();

            if (viewTitle == null) {
                viewTitle = viewId;
            }

            appContext = context.getApplicationContext();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_TRACK_VIEW);
            intent.putExtra(KEY_VIEW_ID, viewId);
            intent.putExtra(KEY_VIEW_TITLE, viewTitle);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    /**
     * Call this whenever you display a new view. Use this in views where you are tracking position.
     *  If the tracker has not been
     * initialized, this call will be ignored.
     *
     * @param context
     *            the context of the current view being tracked
     *            
     * @param viewId
     *            the id of the view being displayed. Must not be null.
     * @param viewTitle
     *            the title of the view. may be null.
     * @param scrollPositionTop
     *            Scroll Position Top
     * @param scrollWindowHeight
     *            Scroll Window Height
     * @param totalContentHeight
     *            Total Content Height
     * @param fullyRenderedDocWidth
     *            Width of the document fully rendered
     */
    public static void trackView(Context context, String viewId, String viewTitle,
                                 int scrollPositionTop, int scrollWindowHeight,
                                 int totalContentHeight, int fullyRenderedDocWidth) {
        try {
            didInit();

            if (context == null) {
                throw new NullPointerException("context cannot be null");
            }

            if (viewId == null) {
                throw new NullPointerException("viewId cannot be null");
            }

            resetUserInteractionMonitor();

            if (TextUtils.isEmpty(viewTitle)) {
                viewTitle = viewId;
            }

            appContext = context.getApplicationContext();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_TRACK_VIEW);
            intent.putExtra(KEY_VIEW_ID, viewId);
            intent.putExtra(KEY_VIEW_TITLE, viewTitle);
            intent.putExtra(KEY_POSITION_TOP, scrollPositionTop);
            intent.putExtra(KEY_WINDOW_HEIGHT, scrollWindowHeight);
            intent.putExtra(KEY_CONTENT_HEIGHT, totalContentHeight);
            intent.putExtra(KEY_DOC_WIDTH, fullyRenderedDocWidth);

            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    private static void resetUserInteractionMonitor() {
        if (userInteractSubscription != null && !userInteractSubscription.isUnsubscribed()) {
            userInteractSubscription.unsubscribe();
        }
    }

    /**
     * Call this whenever the user leaves an activity. This will be used as a
     * hint that the user might have left the app. If the tracker has not been
     * initialized, this call will be ignored.
     */
    public static void userLeftView(String viewId) {
        try {
            didInit();
            didStartTracking();

            if (viewId == null) {
                throw new NullPointerException("viewId cannot be null");
            }

            resetUserInteractionMonitor();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_LEFT_VIEW);
            intent.putExtra(KEY_VIEW_ID, viewId);

            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    /**
     * Call this whenever the user interacts with your app. If the tracker has
     * not been initialized, this call will be ignored. You will likely want to
     * put this in your onUserInteraction() function of your activity.
     */
    public static void userInteracted() {
        try {
            didInit();
            didStartTracking();
            startUserInteractTimer();
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    /**
     * Perform interaction service start on a rolling time window
     */
    private static void startUserInteractTimer() {
        try {
            if (userInteractSubscription != null && !userInteractSubscription.isUnsubscribed()) {
                return;
            }

            Intent intent = new Intent(appContext, ChartbeatService.class);
            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_USER_INTERACTED);
            sendServiceSignal(intent);

            userInteractSubscription = Observable.timer(USER_INTERACT_WINDOW_IN_MILLISECONDS, TimeUnit.MILLISECONDS)
                    .observeOn(Schedulers.io())
                    .subscribe(new Subscriber<Long>() {
                        @Override
                        public void onCompleted() {
                            // Intentionally left blank, resets timer, automatically unsubscribe
                        }

                        @Override
                        public void onError(Throwable e) {
                            Logger.e(TAG, e.getMessage());
                        }

                        @Override
                        public void onNext(Long aLong) {

                        }
                    });
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this whenever the user is writing/typing. If the tracker has not
     * been initialized, this call will be ignored.
     */
    public static void userTyped() {
        try {
            didInit();
            didStartTracking();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_USER_TYPED);

            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    /**
     * Call this method to set the domain for the current view. This will only need be done if
     * the app needs to send data to different dashboards depending on the view. See our main
     * documentation about domain/subdomain settings for clarification, if you think you may need
     * to use this setting.
     *
     * @param domain
     *            the domain name that the current view should track under.
     */
    public static void setDomain(String domain) {
        try {
            didInit();
            didStartTracking();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_DOMAIN);
            intent.putExtra(KEY_DOMAIN, domain);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this method to set the subdomain for the current view. This will not
     * be necessary in most situations since the subdomain will automatically default
     * to the domain setting. See our main documentation about domain/subdomain settings
     * for clarification, if you think you may need to use this setting.
     *
     * @param subdomain
     *            the subdomain name that the current view should track under.
     */
    public static void setSubdomain(String subdomain) {
        try {
            didInit();
            didStartTracking();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_SUBDOMAIN);
            intent.putExtra(KEY_SUBDOMAIN, subdomain);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this method to set the zone(s) for the current view. This data will
     * be purged when changing the view, so be sure to call this after
     * calling trackView().
     *
     * @param zones
     *            a comma-delimited list of zones.
     */
    public static void setZones(String zones) {
        try {
            didInit();
            didStartTracking();

            setZonesImpl(zones);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this method to set the zone(s) for the current view. Note that any
     * commas found in the zone strings will be removed because that is the
     * delimiter.
     *
     * @param zones
     */
    public static void setZones(Collection<String> zones) {
        try {
            didInit();
            didStartTracking();

            setZonesImpl(StringUtils.collectionToCommaString(zones));
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    private static void setZonesImpl(String zones) {
        try {
            didInit();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_ZONES);
            intent.putExtra(KEY_ZONES, zones);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this method to set the user as a paid subscriber.
     * This should be called once per execution of the app or
     * whenever the user's state changes.
     *
     */
    public static void setUserPaid() {
        setUserSubscriptionImpl(SubscriptionState.PAID);
    }

    /**
     * Call this method to set the user as a logged in user.
     * This should be called once per execution of the app or
     * whenever the user's state changes.
     */

     public static void setUserLoggedIn() {
        setUserSubscriptionImpl(SubscriptionState.LOGGED_IN);
     }

     /**
     * Call this method to set the user as an anonymous user.
     * This should be called once per execution of the app or
     * whenever the user's state changes.
     */
     public static void setUserAnonymous() {
        setUserSubscriptionImpl(SubscriptionState.ANONYMOUS);
     }

    private static void setUserSubscriptionImpl(SubscriptionState subState) {
        try {
            didInit();
            didStartTracking();
            Intent intent = new Intent(appContext, ChartbeatService.class);
            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_SUBSCRIPTION_STATE);
            intent.putExtra(KEY_SUBSCRIPTION_STATE, subState);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }


    /**
     * Call this method to set the author(s) for the current view. This data
     * will be purged when changing the view, so be sure to call this after
     * calling trackView().
     *
     * @param authors
     *            a comma-delimited list of authors.
     */
    public static void setAuthors(String authors) {
        try {
            didInit();
            didStartTracking();

            setAuthorsImpl(authors);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this method to set the authors(s) for the current view. Note that
     * any commas found in the author strings will be removed because that is
     * the delimiter.
     *
     * @param authors
     */
    public static void setAuthors(Collection<String> authors) {
        try {
            didInit();
            didStartTracking();

            setAuthorsImpl(StringUtils.collectionToCommaString(authors));
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    private static void setAuthorsImpl(String authors) {
        try {
            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_AUTHORS);
            intent.putExtra(KEY_AUTHORS, authors);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    /**
     * Call this method to set the section(s) for the current view. This data
     * will be purged when changing the view, so be sure to call this after
     * calling trackView().
     *
     * @param sections
     *            a comma-delimited list of sections.
     */
    public static void setSections(String sections) {
        try {
            didInit();
            didStartTracking();

            setSectionsImpl(sections);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this method to set the sections(s) for the current view. Note that
     * any commas found in the section strings will be removed because that is
     * the delimiter.
     *
     * @param sections
     */
    public static void setSections(Collection<String> sections) {
        try {
            didInit();
            didStartTracking();

            setSectionsImpl(StringUtils.collectionToCommaString(sections));
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    private static void setSectionsImpl(String sections) {
        try {
            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_SECTIONS);
            intent.putExtra(KEY_SECTIONS, sections);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * Call this to set the load time of the current page/view. This data will
     * be purged when changing the view, so be sure to call this after
     * calling trackView().
     * */
    public static void setViewLoadTime(float pageLoadTime) {
        try {
            didInit();
            didStartTracking();

            if (pageLoadTime < 0.0f) {
                Logger.e(TAG, "Page load time cannot be negative");
                return;
            }

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_VIEW_LOADING_TIME);
            intent.putExtra(KEY_VIEW_LOADING_TIME, pageLoadTime);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }
    }

    /**
     * sets the position of the current view, assuming it scrolls. If it does
     * not scroll, don't call this function. Negative values will not be passed
     * to the server.
     *
     * @param scrollPositionTop
     *            Scroll Position Top
     * @param scrollWindowHeight
     *            Scroll Window Height
     * @param totalContentHeight
     *            Total Content Height
     * @param fullyRenderedDocWidth
     *            Width of the document fully rendered
     */
    public static void setPosition(int scrollPositionTop, int scrollWindowHeight, int totalContentHeight, int fullyRenderedDocWidth) {
        try {
            didInit();
            didStartTracking();

            Intent intent = new Intent(appContext, ChartbeatService.class);

            intent.putExtra(KEY_SDK_ACTION_TYPE, ACTION_SET_POSITION);

            intent.putExtra(KEY_POSITION_TOP, scrollPositionTop);
            intent.putExtra(KEY_WINDOW_HEIGHT, scrollWindowHeight);
            intent.putExtra(KEY_CONTENT_HEIGHT, totalContentHeight);
            intent.putExtra(KEY_DOC_WIDTH, fullyRenderedDocWidth);
            sendServiceSignal(intent);
        } catch (Exception e) {
            AwsLogger.getInstance().logError(e);
        }

    }

    public static void didInit() {
        if (appContext == null && TextUtils.isEmpty(accountID)) {
            throw new IllegalStateException("Chartbeat: SDK has not been initialized with an Account ID");
        }
    }

    public static void didStartTracking() {
        if (appContext == null) {
            throw new IllegalStateException("Chartbeat: View tracking hasn't started, please call Tracker.trackView() in onResume() first");
        }
    }

    private static void sendServiceSignal(Intent intent) {
        try {
            appContext.startService(intent);
        } catch (Exception e) {
            // Preventive measure for Android Oreo background limits.
            AwsLogger.getInstance().logError(e);        }
    }
}
