package com.chartbeat.androidsdk;

import android.content.Context;
import android.content.SharedPreferences;

import java.util.Date;


final class SiteVisitTracker {
    private static final String TAG = "Chartbeat SiteVisitTracker";
    private static final int MINUTES_TO_TRACK_SITE_VISIT = 30;
    private static final int SITE_VISIT_ID_CHAR_LENGTH = 28;

    private static final String KEY_LAST_SITE_VISIT_ID = "site-visit-uid-";
    private static final String KEY_LAST_SITE_VISIT_REFRESH_TIME = "site-visit-refresh_time-";
    private static final String KEY_LAST_SITE_VISIT_DEPTH = "site-visit-depth-";
    private static final String KEY_LAST_SITE_VISIT_REFERRER = "site-visit-referrer-";

    protected SharedPreferences prefs;

    SiteVisitTracker(Context context) {
        if (context == null) {
            throw new NullPointerException("Activity or application context cannot be null");
        }

        this.prefs = context.getSharedPreferences(ChartBeatTracker.CHARTBEAT_PREFS, Context.MODE_PRIVATE);
    }

    public String getSiteVisitId() {
        return prefs.getString(KEY_LAST_SITE_VISIT_ID, null);
    }

    public void storeSiteVisitId(String siteVisitID) {
        SharedPreferences.Editor editor = prefs.edit();
        editor.putString(KEY_LAST_SITE_VISIT_ID, siteVisitID);
        editor.apply();
    }

    public int getSiteVisitDepth() {
        return prefs.getInt(KEY_LAST_SITE_VISIT_DEPTH, 0);
    }

    public void visited() {
        storeSiteVisitRefreshTime(new Date());
    }

    public String getSiteVisitReferrer() {
        return prefs.getString(KEY_LAST_SITE_VISIT_REFERRER, null);
    }

    public void trackSiteVisit(String newSiteReferrer) {
        if (newSiteReferrer == null) {
            newSiteReferrer = new String();
        }

        String siteVisitReferrer = getSiteVisitReferrer();
        Boolean shouldResetSiteVisit = (siteVisitReferrer != null && newSiteReferrer.length() > 0 && !siteVisitReferrer.equals(newSiteReferrer)) || hasSiteVisitExpired() || (siteVisitReferrer == null && newSiteReferrer.length() > 0);


        if (!shouldResetSiteVisit) {
            storeSiteVisitDepth(getSiteVisitDepth()+1);
        } else {
            storeSiteVisitDepth(1);
            storeSiteVisitReferrer(newSiteReferrer);
            storeSiteVisitId(SecurityUtils.randomChars(SITE_VISIT_ID_CHAR_LENGTH));
            storeSiteVisitRefreshTime(new Date());
        }
    }

    void storeSiteVisitDepth(int siteVisitDepth) {
        SharedPreferences.Editor editor = prefs.edit();
        editor.putInt(KEY_LAST_SITE_VISIT_DEPTH, siteVisitDepth);
        editor.apply();
    }

    void storeSiteVisitReferrer(String siteVisitReferrer) {
        SharedPreferences.Editor editor = prefs.edit();
        editor.putString(KEY_LAST_SITE_VISIT_REFERRER, siteVisitReferrer);
        editor.apply();
    }

    Date getSiteVisitRefreshTime() {
        Long dateLong = prefs.getLong(KEY_LAST_SITE_VISIT_REFRESH_TIME, 0);
        if (dateLong != 0) {
            return new Date(dateLong);
        } else {
            return null;
        }
    }

    void storeSiteVisitRefreshTime(Date siteVisitRefreshTime) {
        SharedPreferences.Editor editor = prefs.edit();
        editor.putLong(KEY_LAST_SITE_VISIT_REFRESH_TIME, siteVisitRefreshTime.getTime());
        editor.apply();
    }

    Boolean hasSiteVisitExpired() {
        Date rightNow = new Date();
        Date siteVisitExpirationTime = getSiteVisitRefreshTime();

        if (siteVisitExpirationTime == null) {
            return true; //no existing site visit. consider it expired.
        }

        siteVisitExpirationTime.setMinutes(siteVisitExpirationTime.getMinutes() + MINUTES_TO_TRACK_SITE_VISIT);
        return siteVisitExpirationTime.before(rightNow);
    }
}
