/*
Copyright 2007 Infordata S.p.A.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/*
!!V 10/04/97 rel. 0.93 - moved to crt5250 package.
    16/04/97 rel. 0.94 - toPacked method added.
    28/04/97 rel. 0.94c- corrected problem with underscore.
    27/05/97 rel. 1.00 - first release.
    25/07/97 rel. 1.03a- '|' char mapping.
    03/03/98 rel. _.___- SWING and reorganization.
    ***
    30/06/98 rel. _.___- Swing, JBuilder2 e VSS.
 */

package net.infordata.em.crt5250;

/**
 * http://www-03.ibm.com/systems/i/software/globalization/ccsid.html
 * http://www-306.ibm.com/software/globalization/ccsid/ccsid_registered.jsp
 *
 * @author Valentino Proietti - Infordata S.p.A.
 */
public class XIEbcdicNTranslator extends XIEbcdicTranslator {

  private final int[] ivEbcdicToChar;
  private final int[] ivCharToEbcdic = new int[65536];

  protected XIEbcdicNTranslator(int[] ebcdic) {
    ivEbcdicToChar = ebcdic;
    for (int i = ivEbcdicToChar.length - 1; i >= 0; i--) {
      ivCharToEbcdic[ivEbcdicToChar[i]] = i;
    }
  }

  /**
   * From ebcdic code to char
   *
   * @param aEbcdicChar ebcdic code to get the char from
   * @return ebcdic code obtained from the given char
   */
  @Override
  public char toChar(byte aEbcdicChar) {
    return (char) ivEbcdicToChar[toInt(aEbcdicChar)];
  }

  /**
   * From char to ebcdic code
   *
   * @param aChar char to get the ebcdic code from
   * @return char obtained from the given ebcdic code
   */
  @Override
  public byte toEBCDIC(char aChar) {
    return (byte) ivCharToEbcdic[
        Math.max(0, Math.min(ivCharToEbcdic.length - 1, (int) aChar))];
  }

  private static class Euro extends XIEbcdicNTranslator {

    private final byte ivEbcdicEuro;

    protected Euro(int[] ebcdic, byte ebcdicEuro) {
      super(ebcdic);
      ivEbcdicEuro = ebcdicEuro;
    }

    @Override
    public char toChar(byte aEbcdicChar) {
      if (aEbcdicChar == ivEbcdicEuro) {
        return '�';
      }
      return super.toChar(aEbcdicChar);
    }

    @Override
    public byte toEBCDIC(char aChar) {
      if (aChar == '�') {
        return ivEbcdicEuro;
      }
      return super.toEBCDIC(aChar);
    }
  }

  private static final int[] CP37 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 224, 225, 227, 229, /*  64 -  71  */
      231, 241, 162, 46, 60, 40, 43, 124, /*  72 -  79  */
      38, 233, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 223, 33, 36, 42, 41, 59, 172, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 197, /*  96 - 103  */
      199, 209, 166, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 96, 58, 35, 64, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      181, 126, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      94, 163, 165, 183, 169, 167, 182, 188, /* 176 - 183  */
      189, 190, 91, 93, 175, 168, 180, 215, /* 184 - 191  */
      123, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 242, 243, 245, /* 200 - 207  */
      125, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 252, 249, 250, 255, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP273 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 123, 224, 225, 227, 229, /*  64 -  71  */
      231, 241, 196, 46, 60, 40, 43, 33, /*  72 -  79  */
      38, 233, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 126, 220, 36, 42, 41, 59, 94, /*  88 -  95  */
      45, 47, 194, 91, 192, 193, 195, 197, /*  96 - 103  */
      199, 209, 246, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 96, 58, 35, 167, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      181, 223, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 163, 165, 183, 169, 64, 182, 188, /* 176 - 183  */
      189, 190, 172, 124, 175, 168, 180, 215, /* 184 - 191  */
      228, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 166, 242, 243, 245, /* 200 - 207  */
      252, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 125, 249, 250, 255, /* 216 - 223  */
      214, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 92, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 93, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP277 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 224, 225, 227, 125, /*  64 -  71  */
      231, 241, 35, 46, 60, 40, 43, 33, /*  72 -  79  */
      38, 233, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 223, 164, 197, 42, 41, 59, 94, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 36, /*  96 - 103  */
      199, 209, 248, 44, 37, 95, 62, 63, /* 104 - 111  */
      166, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 96, 58, 198, 216, 39, 61, 34, /* 120 - 127  */
      64, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 123, 184, 91, 93, /* 152 - 159  */
      181, 252, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 163, 165, 183, 169, 167, 182, 188, /* 176 - 183  */
      189, 190, 172, 124, 175, 168, 180, 215, /* 184 - 191  */
      230, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 242, 243, 245, /* 200 - 207  */
      229, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 126, 249, 250, 255, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP278 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 123, 224, 225, 227, 125, /*  64 -  71  */
      231, 241, 167, 46, 60, 40, 43, 33, /*  72 -  79  */
      38, 96, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 223, 164, 197, 42, 41, 59, 94, /*  88 -  95  */
      45, 47, 194, 35, 192, 193, 195, 36, /*  96 - 103  */
      199, 209, 246, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 233, 58, 196, 214, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 93, /* 152 - 159  */
      181, 252, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 163, 165, 183, 169, 91, 182, 188, /* 176 - 183  */
      189, 190, 172, 124, 175, 168, 180, 215, /* 184 - 191  */
      228, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 166, 242, 243, 245, /* 200 - 207  */
      229, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 126, 249, 250, 255, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 64, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP280 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 123, 225, 227, 229, /*  64 -  71  */
      92, 241, 176, 46, 60, 40, 43, 33, /*  72 -  79  */
      38, 93, 234, 235, 125, 237, 238, 239, /*  80 -  87  */
      126, 223, 233, 36, 42, 41, 59, 94, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 197, /*  96 - 103  */
      199, 209, 242, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 249, 58, 163, 167, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      91, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      181, 236, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 35, 165, 183, 169, 64, 182, 188, /* 176 - 183  */
      189, 190, 172, 124, 175, 168, 180, 215, /* 184 - 191  */
      224, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 166, 243, 245, /* 200 - 207  */
      232, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 252, 96, 250, 255, /* 216 - 223  */
      231, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP284 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 224, 225, 227, 229, /*  64 -  71  */
      231, 166, 91, 46, 60, 40, 43, 124, /*  72 -  79  */
      38, 233, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 223, 93, 36, 42, 41, 59, 172, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 197, /*  96 - 103  */
      199, 35, 241, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 96, 58, 209, 64, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      181, 168, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 163, 165, 183, 169, 167, 182, 188, /* 176 - 183  */
      189, 190, 94, 33, 175, 126, 180, 215, /* 184 - 191  */
      123, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 242, 243, 245, /* 200 - 207  */
      125, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 252, 249, 250, 255, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP285 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 224, 225, 227, 229, /*  64 -  71  */
      231, 241, 36, 46, 60, 40, 43, 124, /*  72 -  79  */
      38, 233, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 223, 33, 163, 42, 41, 59, 172, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 197, /*  96 - 103  */
      199, 209, 166, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 96, 58, 35, 64, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      181, 175, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 91, 165, 183, 169, 167, 182, 188, /* 176 - 183  */
      189, 190, 94, 93, 126, 168, 180, 215, /* 184 - 191  */
      123, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 242, 243, 245, /* 200 - 207  */
      125, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 252, 249, 250, 255, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP297 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 64, 225, 227, 229, /*  64 -  71  */
      92, 241, 176, 46, 60, 40, 43, 33, /*  72 -  79  */
      38, 123, 234, 235, 125, 237, 238, 239, /*  80 -  87  */
      236, 223, 167, 36, 42, 41, 59, 94, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 197, /*  96 - 103  */
      199, 209, 249, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 181, 58, 163, 224, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      91, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      96, 168, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 35, 165, 183, 169, 93, 182, 188, /* 176 - 183  */
      189, 190, 172, 124, 175, 126, 180, 215, /* 184 - 191  */
      233, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 242, 243, 245, /* 200 - 207  */
      232, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 252, 166, 250, 255, /* 216 - 223  */
      231, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP424 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 193, 194, 195, 196, 197, 198, 199, /*  64 -  71  */
      200, 201, 162, 46, 60, 40, 43, 124, /*  72 -  79  */
      38, 216, 217, 226, 227, 228, 229, 230, /*  80 -  87  */
      231, 232, 33, 36, 42, 41, 59, 172, /*  88 -  95  */
      45, 47, 233, 192, 186, 161, 210, 191, /*  96 - 103  */
      212, 213, 166, 44, 37, 95, 62, 63, /* 104 - 111  */
      242, 248, 249, 209, 160, 255, 234, 241, /* 112 - 119  */
      177, 96, 58, 35, 64, 39, 61, 34, /* 120 - 127  */
      214, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 246, 211, 243, 208, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 218, 240, 225, 184, 245, 164, /* 152 - 159  */
      181, 126, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 170, 224, 244, 202, 175, 174, /* 168 - 175  */
      94, 163, 165, 183, 169, 167, 182, 188, /* 176 - 183  */
      189, 190, 91, 93, 250, 168, 180, 215, /* 184 - 191  */
      123, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 203, 204, 205, 206, 207, /* 200 - 207  */
      125, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 219, 220, 221, 222, 223, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 235, 236, 237, 238, 239, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 251, 252, 253, 254, 0, /* 248 - 255  */
  };

  private static final int[] CP500 = {
      0, 1, 2, 3, 156, 9, 134, 127, /*   0 -   7  */
      151, 141, 142, 11, 12, 13, 14, 15, /*   8 -  15  */
      16, 17, 18, 19, 157, 133, 8, 135, /*  16 -  23  */
      24, 25, 146, 143, 28, 29, 30, 31, /*  24 -  31  */
      128, 129, 130, 131, 132, 10, 23, 27, /*  32 -  39  */
      136, 137, 138, 139, 140, 5, 6, 7, /*  40 -  47  */
      144, 145, 22, 147, 148, 149, 150, 4, /*  48 -  55  */
      152, 153, 154, 155, 20, 21, 158, 26, /*  56 -  63  */
      32, 160, 226, 228, 224, 225, 227, 229, /*  64 -  71  */
      231, 241, 91, 46, 60, 40, 43, 33, /*  72 -  79  */
      38, 233, 234, 235, 232, 237, 238, 239, /*  80 -  87  */
      236, 223, 93, 36, 42, 41, 59, 94, /*  88 -  95  */
      45, 47, 194, 196, 192, 193, 195, 197, /*  96 - 103  */
      199, 209, 166, 44, 37, 95, 62, 63, /* 104 - 111  */
      248, 201, 202, 203, 200, 205, 206, 207, /* 112 - 119  */
      204, 96, 58, 35, 64, 39, 61, 34, /* 120 - 127  */
      216, 97, 98, 99, 100, 101, 102, 103, /* 128 - 135  */
      104, 105, 171, 187, 240, 253, 254, 177, /* 136 - 143  */
      176, 106, 107, 108, 109, 110, 111, 112, /* 144 - 151  */
      113, 114, 170, 186, 230, 184, 198, 164, /* 152 - 159  */
      181, 126, 115, 116, 117, 118, 119, 120, /* 160 - 167  */
      121, 122, 161, 191, 208, 221, 222, 174, /* 168 - 175  */
      162, 163, 165, 183, 169, 167, 182, 188, /* 176 - 183  */
      189, 190, 172, 124, 175, 168, 180, 215, /* 184 - 191  */
      123, 65, 66, 67, 68, 69, 70, 71, /* 192 - 199  */
      72, 73, 173, 244, 246, 242, 243, 245, /* 200 - 207  */
      125, 74, 75, 76, 77, 78, 79, 80, /* 208 - 215  */
      81, 82, 185, 251, 252, 249, 250, 255, /* 216 - 223  */
      92, 247, 83, 84, 85, 86, 87, 88, /* 224 - 231  */
      89, 90, 178, 212, 214, 210, 211, 213, /* 232 - 239  */
      48, 49, 50, 51, 52, 53, 54, 55, /* 240 - 247  */
      56, 57, 179, 219, 220, 217, 218, 0, /* 248 - 255  */
  };

  private static final int[] CP850 = {
      0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
      '\b', '\t', '\n', 0x000B, '\f', '\r', 0x000E, 0x000F,
      0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
      0x0018, 0x0019, 0x001C, 0x001B, 0x007F, 0x001D, 0x001E, 0x001F,
      0x0020, 0x0021, '\"', 0x0023, 0x0024, 0x0025, 0x0026, '\'',
      0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
      0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
      0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
      0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
      0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
      0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
      0x0058, 0x0059, 0x005A, 0x005B, '\\', 0x005D, 0x005E, 0x005F,
      0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
      0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
      0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
      0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0x001A,
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7,
      0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5,
      0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9,
      0x00FF, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x00D7, 0x0192,
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA,
      0x00BF, 0x00AE, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB,
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00C1, 0x00C2, 0x00C0,
      0x00A9, 0x2563, 0x2551, 0x2557, 0x255D, 0x00A2, 0x00A5, 0x2510,
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x00E3, 0x00C3,
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x00A4,
      0x00F0, 0x00D0, 0x00CA, 0x00CB, 0x00C8, 0x0131, 0x00CD, 0x00CE,
      0x00CF, 0x2518, 0x250C, 0x2588, 0x2584, 0x00A6, 0x00CC, 0x2580,
      0x00D3, 0x00DF, 0x00D4, 0x00D2, 0x00F5, 0x00D5, 0x00B5, 0x00FE,
      0x00DE, 0x00DA, 0x00DB, 0x00D9, 0x00FD, 0x00DD, 0x00AF, 0x00B4,
      0x00AD, 0x00B1, 0x2017, 0x00BE, 0x00B6, 0x00A7, 0x00F7, 0x00B8,
      0x00B0, 0x00A8, 0x00B7, 0x00B9, 0x00B3, 0x00B2, 0x25A0, 0x00A0,
  };

  private static final int[] CP870 = {
      0x0000, 0x0001, 0x0002, 0x0003, 0x009C, '\t', 0x0086, 0x007F,
      0x0097, 0x008D, 0x008E, 0x000B, '\f', '\r', 0x000E, 0x000F,
      0x0010, 0x0011, 0x0012, 0x0013, 0x009D, 0x0085, '\b', 0x0087,
      0x0018, 0x0019, 0x0092, 0x008F, 0x001C, 0x001D, 0x001E, 0x001F,
      0x0080, 0x0081, 0x0082, 0x0083, 0x0084, '\n', 0x0017, 0x001B,
      0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x0005, 0x0006, 0x0007,
      0x0090, 0x0091, 0x0016, 0x0093, 0x0094, 0x0095, 0x0096, 0x0004,
      0x0098, 0x0099, 0x009A, 0x009B, 0x0014, 0x0015, 0x009E, 0x001A,
      0x0020, 0x00A0, 0x00E2, 0x00E4, 0x0163, 0x00E1, 0x0103, 0x010D,
      0x00E7, 0x0107, 0x005B, 0x002E, 0x003C, 0x0028, 0x002B, 0x0021,
      0x0026, 0x00E9, 0x0119, 0x00EB, 0x016F, 0x00ED, 0x00EE, 0x013E,
      0x013A, 0x00DF, 0x005D, 0x0024, 0x002A, 0x0029, 0x003B, 0x005E,
      0x002D, 0x002F, 0x00C2, 0x00C4, 0x02DD, 0x00C1, 0x0102, 0x010C,
      0x00C7, 0x0106, 0x007C, 0x002C, 0x0025, 0x005F, 0x003E, 0x003F,
      0x02C7, 0x00C9, 0x0118, 0x00CB, 0x016E, 0x00CD, 0x00CE, 0x013D,
      0x0139, 0x0060, 0x003A, 0x0023, 0x0040, '\'', 0x003D, '\"',
      0x02D8, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
      0x0068, 0x0069, 0x015B, 0x0148, 0x0111, 0x00FD, 0x0159, 0x015F,
      0x00B0, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F, 0x0070,
      0x0071, 0x0072, 0x0142, 0x0144, 0x0161, 0x00B8, 0x02DB, 0x00A4,
      0x0105, 0x007E, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078,
      0x0079, 0x007A, 0x015A, 0x0147, 0x0110, 0x00DD, 0x0158, 0x015E,
      0x02D9, 0x0104, 0x017C, 0x0162, 0x017B, 0x00A7, 0x017E, 0x017A,
      0x017D, 0x0179, 0x0141, 0x0143, 0x0160, 0x00A8, 0x00B4, 0x00D7,
      0x007B, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
      0x0048, 0x0049, 0x00AD, 0x00F4, 0x00F6, 0x0155, 0x00F3, 0x0151,
      0x007D, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F, 0x0050,
      0x0051, 0x0052, 0x011A, 0x0171, 0x00FC, 0x0165, 0x00FA, 0x011B,
      '\\', 0x00F7, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058,
      0x0059, 0x005A, 0x010F, 0x00D4, 0x00D6, 0x0154, 0x00D3, 0x0150,
      0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
      0x0038, 0x0039, 0x010E, 0x0170, 0x00DC, 0x0164, 0x00DA, 0x009F,
  };

  private static final int[] CP838 = {
      0x0000, 0x0001, 0x0002, 0x0003, 0x009C, '\t', 0x0086, 0x007F,
      0x0097, 0x008D, 0x008E, 0x000B, '\f', '\r', 0x000E, 0x000F,
      0x0010, 0x0011, 0x0012, 0x0013, 0x009D, 0x0085, '\b', 0x0087,
      0x0018, 0x0019, 0x0092, 0x008F, 0x001C, 0x001D, 0x001E, 0x001F,
      0x0080, 0x0081, 0x0082, 0x0083, 0x0084, '\n', 0x0017, 0x001B,
      0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x0005, 0x0006, 0x0007,
      0x0090, 0x0091, 0x0016, 0x0093, 0x0094, 0x0095, 0x0096, 0x0004,
      0x0098, 0x0099, 0x009A, 0x009B, 0x0014, 0x0015, 0x009E, 0x001A,
      0x0020, 0x00A0, 0x0E01, 0x0E02, 0x0E03, 0x0E04, 0x0E05, 0x0E06,
      0x0E07, 0x005B, 0x00A2, 0x002E, 0x003C, 0x0028, 0x002B, 0x007C,
      0x0026, 0x0E48, 0x0E08, 0x0E09, 0x0E0A, 0x0E0B, 0x0E0C, 0x0E0D,
      0x0E0E, 0x005D, 0x0021, 0x0024, 0x002A, 0x0029, 0x003B, 0x00AC,
      0x002D, 0x002F, 0x0E0F, 0x0E10, 0x0E11, 0x0E12, 0x0E13, 0x0E14,
      0x0E15, 0x005E, 0x00A6, 0x002C, 0x0025, 0x005F, 0x003E, 0x003F,
      0x0E3F, 0x0E4E, 0x0E16, 0x0E17, 0x0E18, 0x0E19, 0x0E1A, 0x0E1B,
      0x0E1C, 0x0060, 0x003A, 0x0023, 0x0040, '\'', 0x003D, '\"',
      0x0E4F, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
      0x0068, 0x0069, 0x0E1D, 0x0E1E, 0x0E1F, 0x0E20, 0x0E21, 0x0E22,
      0x0E5A, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F, 0x0070,
      0x0071, 0x0072, 0x0E23, 0x0E24, 0x0E25, 0x0E26, 0x0E27, 0x0E28,
      0x0E5B, 0x007E, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078,
      0x0079, 0x007A, 0x0E29, 0x0E2A, 0x0E2B, 0x0E2C, 0x0E2D, 0x0E2E,
      0x0E50, 0x0E51, 0x0E52, 0x0E53, 0x0E54, 0x0E55, 0x0E56, 0x0E57,
      0x0E58, 0x0E59, 0x0E2F, 0x0E30, 0x0E31, 0x0E32, 0x0E33, 0x0E34,
      0x007B, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
      0x0048, 0x0049, 0x0E49, 0x0E35, 0x0E36, 0x0E37, 0x0E38, 0x0E39,
      0x007D, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F, 0x0050,
      0x0051, 0x0052, 0x0E3A, 0x0E40, 0x0E41, 0x0E42, 0x0E43, 0x0E44,
      '\\', 0x0E4A, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058,
      0x0059, 0x005A, 0x0E45, 0x0E46, 0x0E47, 0x0E48, 0x0E49, 0x0E4A,
      0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
      0x0038, 0x0039, 0x0E4B, 0x0E4C, 0x0E4D, 0x0E4B, 0x0E4C, 0x009F
  };

  public static final XIEbcdicNTranslator TRANSLATOR_CP37 =
      new XIEbcdicNTranslator(CP37);
  public static final XIEbcdicNTranslator TRANSLATOR_CP273 =
      new XIEbcdicNTranslator(CP273);
  public static final XIEbcdicNTranslator TRANSLATOR_CP277 =
      new XIEbcdicNTranslator(CP277);
  public static final XIEbcdicNTranslator TRANSLATOR_CP278 =
      new XIEbcdicNTranslator(CP278);
  public static final XIEbcdicNTranslator TRANSLATOR_CP280 =
      new XIEbcdicNTranslator(CP280);
  public static final XIEbcdicNTranslator TRANSLATOR_CP284 =
      new XIEbcdicNTranslator(CP284);
  public static final XIEbcdicNTranslator TRANSLATOR_CP285 =
      new XIEbcdicNTranslator(CP285);
  public static final XIEbcdicNTranslator TRANSLATOR_CP297 =
      new XIEbcdicNTranslator(CP297);
  public static final XIEbcdicNTranslator TRANSLATOR_CP424 =
      new XIEbcdicNTranslator(CP424);
  public static final XIEbcdicNTranslator TRANSLATOR_CP500 =
      new XIEbcdicNTranslator(CP500);
  public static final XIEbcdicNTranslator TRANSLATOR_CP850 =
      new XIEbcdicNTranslator(CP850);
  public static final XIEbcdicNTranslator TRANSLATOR_CP870 =
      new XIEbcdicNTranslator(CP870);
  public static final XIEbcdicNTranslator TRANSLATOR_CP838 =
      new XIEbcdicNTranslator(CP838);
  // Derived with
  public static final XIEbcdicNTranslator TRANSLATOR_CP1140 =
      new XIEbcdicNTranslator.Euro(CP37, (byte) 159);
  public static final XIEbcdicNTranslator TRANSLATOR_CP1141 =
      new XIEbcdicNTranslator.Euro(CP273, (byte) 159);
  public static final XIEbcdicNTranslator TRANSLATOR_CP1144 =
      new XIEbcdicNTranslator.Euro(CP280, (byte) 159);
  public static final XIEbcdicNTranslator TRANSLATOR_CP1147 =
      new XIEbcdicNTranslator.Euro(CP297, (byte) 159);
  public static final XIEbcdicNTranslator TRANSLATOR_CP1153 =
      new XIEbcdicNTranslator.Euro(CP870, (byte) 159);
  public static final XIEbcdicNTranslator TRANSLATOR_CP1160 =
      new XIEbcdicNTranslator.Euro(CP838, (byte) 0xFE);

  private static int[] SCP2CP(String scp) {
    if (scp.length() != 256) {
      throw new IllegalArgumentException();
    }
    int[] res = new int[scp.length()];
    for (int i = 0; i < res.length; i++) {
      res[i] = scp.charAt(i);
    }
    return res;
  }

  private static final String SCP1025 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0452\u0453\u0451\u0454\u0455\u0456\u0457\u0458\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u0459\u045A\u045B\u045C\u045E\u045F\u042A\u2116\u0402\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u0403\u0401\u0404\u0405\u0406\u0407\u0408\u0409\u007C\u002C\u0025\u005F\u003E\u003F"
          +
          "\u040A\u040B\u040C\u00AD\u040E\u040F\u044E\u0430\u0431\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u0446\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u0434\u0435\u0444\u0433\u0445\u0438"
          +
          "\u0439\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u043A\u043B\u043C\u043D\u043E\u043F"
          +
          "\u044F\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u0440\u0441\u0442\u0443\u0436\u0432"
          +
          "\u044C\u044B\u0437\u0448\u044D\u0449\u0447\u044A\u042E\u0410\u0411\u0426\u0414\u0415\u0424\u0413"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u0425\u0418\u0419\u041A\u041B\u041C"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u041D\u041E\u041F\u042F\u0420\u0421"
          +
          "\\\u00A7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u0422\u0423\u0416\u0412\u042C\u042B"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u0417\u0428\u042D\u0429\u0427\u009F";

  private static final String SCP1026 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u007B\u00F1\u00C7\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u011E\u0130\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u005B\u00D1\u015F\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u0131\u003A\u00D6\u015E\'\u003D\u00DC"
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u007D\u0060\u00A6\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u00A4"
          +
          "\u00B5\u00F6\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u005D\u0024\u0040\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\u00B4\u00D7"
          +
          "\u00E7\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u007E\u00F2\u00F3\u00F5"
          +
          "\u011F\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\\\u00F9\u00FA\u00FF"
          +
          "\u00FC\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u0023\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\"\u00D9\u00DA\u009F";

  private static final String SCP1112 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0161\u00E4\u0105\u012F\u016B\u00E5\u0113\u017E\u00A2\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\u00E9\u0119\u0117\u010D\u0173\u201E\u201C\u0123\u00DF\u0021\u0024\u002A\u0029\u003B\u00AC"
          +
          "\u002D\u002F\u0160\u00C4\u0104\u012E\u016A\u00C5\u0112\u017D\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u0118\u0116\u010C\u0172\u012A\u013B\u0122\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0101\u017C\u0144\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u0156\u0157\u00E6\u0137\u00C6\u00A4"
          +
          "\u00B5\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u201D\u017A\u0100\u017B\u0143\u00AE"
          +
          "\u005E\u00A3\u012B\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u005B\u005D\u0179\u0136\u013C\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u014D\u00F6\u0146\u00F3\u00F5"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u0107\u00FC\u0142\u015B\u2019"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u014C\u00D6\u0145\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u0106\u00DC\u0141\u015A\u009F";

  private static final String SCP1122 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u007B\u00E0\u00E1\u00E3\u007D\u00E7\u00F1\u00A7\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u0060\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u00A4\u00C5\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u0023\u00C0\u00C1\u00C3\u0024\u00C7\u00D1\u00F6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\\\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u00E9\u003A\u00C4\u00D6\'\u003D\"" +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0161\u00FD\u017E\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u005D"
          +
          "\u00B5\u00FC\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u0160\u00DD\u017D\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u005B\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\u00B4\u00D7"
          +
          "\u00E4\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00A6\u00F2\u00F3\u00F5"
          +
          "\u00E5\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u007E\u00F9\u00FA\u00FF"
          +
          "\u00C9\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u0040\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1123 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0452\u0491\u0451\u0454\u0455\u0456\u0457\u0458\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u0459\u045A\u045B\u045C\u045E\u045F\u042A\u2116\u0402\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u0490\u0401\u0404\u0405\u0406\u0407\u0408\u0409\u007C\u002C\u0025\u005F\u003E\u003F"
          +
          "\u040A\u040B\u040C\u00AD\u040E\u040F\u044E\u0430\u0431\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u0446\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u0434\u0435\u0444\u0433\u0445\u0438"
          +
          "\u0439\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u043A\u043B\u043C\u043D\u043E\u043F"
          +
          "\u044F\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u0440\u0441\u0442\u0443\u0436\u0432"
          +
          "\u044C\u044B\u0437\u0448\u044D\u0449\u0447\u044A\u042E\u0410\u0411\u0426\u0414\u0415\u0424\u0413"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u0425\u0418\u0419\u041A\u041B\u041C"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u041D\u041E\u041F\u042F\u0420\u0421"
          +
          "\\\u00A7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u0422\u0423\u0416\u0412\u042C\u042B"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u0417\u0428\u042D\u0429\u0427\u009F";

  private static final String SCP1125 =
      "\u0000\u0001\u0002\u0003\u0004\u0005\u0006\u0007\b\t\n\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001C\u001B\u007F\u001D\u001E\u001F"
          +
          "\u0020\u0021\"\u0023\u0024\u0025\u0026\'\u0028\u0029\u002A\u002B\u002C\u002D\u002E\u002F"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u003A\u003B\u003C\u003D\u003E\u003F"
          +
          "\u0040\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u004A\u004B\u004C\u004D\u004E\u004F"
          +
          "\u0050\u0051\u0052\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u005B\\\u005D\u005E\u005F"
          +
          "\u0060\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u006A\u006B\u006C\u006D\u006E\u006F"
          +
          "\u0070\u0071\u0072\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u007B\u007C\u007D\u007E\u001A"
          +
          "\u0410\u0411\u0412\u0413\u0414\u0415\u0416\u0417\u0418\u0419\u041A\u041B\u041C\u041D\u041E\u041F"
          +
          "\u0420\u0421\u0422\u0423\u0424\u0425\u0426\u0427\u0428\u0429\u042A\u042B\u042C\u042D\u042E\u042F"
          +
          "\u0430\u0431\u0432\u0433\u0434\u0435\u0436\u0437\u0438\u0439\u043A\u043B\u043C\u043D\u043E\u043F"
          +
          "\u2591\u2592\u2593\u2502\u2524\u2561\u2562\u2556\u2555\u2563\u2551\u2557\u255D\u255C\u255B\u2510"
          +
          "\u2514\u2534\u252C\u251C\u2500\u253C\u255E\u255F\u255A\u2554\u2569\u2566\u2560\u2550\u256C\u2567"
          +
          "\u2568\u2564\u2565\u2559\u2558\u2552\u2553\u256B\u256A\u2518\u250C\u2588\u2584\u258C\u2590\u2580"
          +
          "\u0440\u0441\u0442\u0443\u0444\u0445\u0446\u0447\u0448\u0449\u044A\u044B\u044C\u044D\u044E\u044F"
          +
          "\u0401\u0451\u0490\u0491\u0404\u0454\u0406\u0456\u0407\u0457\u00F7\u00B1\u2116\u00A4\u25A0\u00A0";

  private static final String SCP1130 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u0103\u00E5\u00E7\u00F1\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u0303\u00DF\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u0102\u00C5\u00C7\u00D1\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u20AB\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0111\u0309\u0300\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u0152\u00C6\u00A4"
          +
          "\u00B5\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u0110\u0323\u0301\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u0153\u0178\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00F6\u01B0\u00F3\u01A1"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u00D6\u01AF\u00D3\u01A0"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1132 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0E81\u0E82\u0E84\u0E87\u0E88\u0EAA\u0E8A\u005B\u00A2\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\u001A\u0E8D\u0E94\u0E95\u0E96\u0E97\u0E99\u0E9A\u005D\u0021\u0024\u002A\u0029\u003B\u00AC"
          +
          "\u002D\u002F\u0E9B\u0E9C\u0E9D\u0E9E\u0E9F\u0EA1\u0EA2\u005E\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u006B\u001A\u0EA3\u0EA5\u0EA7\u0EAB\u0EAD\u0EAE\u001A\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u001A\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u001A\u001A\u0EAF\u0EB0\u0EB2\u0EB3"
          +
          "\u001A\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u0EB4\u0EB5\u0EB6\u0EB7\u0EB8\u0EB9"
          +
          "\u001A\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u0EBC\u0EB1\u0EBB\u0EBD\u001A\u001A"
          +
          "\u0ED0\u0ED1\u0ED2\u0ED3\u0ED4\u0ED5\u0ED6\u0ED7\u0ED8\u0ED9\u001A\u0EC0\u0EC1\u0EC2\u0EC3\u0EC4"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u001A\u0EC8\u0EC9\u0ECA\u0ECB\u0ECC"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u0ECD\u0EC6\u001A\u0EDC\u0EDD\u001A"
          +
          "\\\u001A\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u001A\u001A\u001A\u001A\u001A\u001A"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u001A\u001A\u001A\u001A\u001A\u009F";

  private static final String SCP1137 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0901\u0902\u0903\u0905\u0906\u0907\u0908\u0909\u090A\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\u090B\u090C\u090D\u090E\u090F\u0910\u0911\u0912\u0913\u0021\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u0914\u0915\u0916\u0917\u0918\u0919\u091A\u091B\u091C\u002C\u0025\u005F\u003E\u003F"
          +
          "\u091D\u091E\u091F\u0920\u0921\u0922\u0923\u0924\u0925\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u0926\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u0927\u0928\u092A\u092B\u092C\u092D"
          +
          "\u092E\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u092F\u0930\u0932\u0933\u0935\u0936"
          +
          "\u200C\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u0937\u0938\u0939\u005B\u093C\u093D"
          +
          "\u093E\u093F\u0940\u0941\u0942\u0943\u0944\u0945\u0946\u0947\u0948\u0949\u094A\u005D\u094B\u094C"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u094D\u0950\u0951\u0952\u001A\u001A"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u0960\u0961\u0962\u0963\u0964\u0965"
          +
          "\\\u200D\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u0966\u0967\u0968\u0969\u096A\u096B"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u096C\u096D\u096E\u096F\u0970\u009F";

  private static final String SCP1142 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u007D\u00E7\u00F1\u0023\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u20AC\u00C5\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u0024\u00C7\u00D1\u00F8\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00A6\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u0060\u003A\u00C6\u00D8\'\u003D\""
          +
          "\u0040\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u00F0\u00FD\u00FE\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u007B\u00B8\u005B\u005D"
          +
          "\u00B5\u00FC\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u00D0\u00DD\u00DE\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\u00B4\u00D7"
          +
          "\u00E6\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00F6\u00F2\u00F3\u00F5"
          +
          "\u00E5\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u007E\u00F9\u00FA\u00FF"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u00D6\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1145 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u00E7\u00A6\u005B\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u005D\u0024\u002A\u0029\u003B\u00AC"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u00C7\u0023\u00F1\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u0060\u003A\u00D1\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u00F0\u00FD\u00FE\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u20AC"
          +
          "\u00B5\u00A8\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u00D0\u00DD\u00DE\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u005E\u0021\u00AF\u007E\u00B4\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00F6\u00F2\u00F3\u00F5"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u00D6\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1146 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u00E7\u00F1\u0024\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u0021\u00A3\u002A\u0029\u003B\u00AC"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u00C7\u00D1\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u00F0\u00FD\u00FE\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u20AC"
          +
          "\u00B5\u00AF\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u00D0\u00DD\u00DE\u00AE"
          +
          "\u00A2\u005B\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u005E\u005D\u007E\u00A8\u00B4\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00F6\u00F2\u00F3\u00F5"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u00D6\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1148 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u00E7\u00F1\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u00C7\u00D1\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u00F0\u00FD\u00FE\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u20AC"
          +
          "\u00B5\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u00D0\u00DD\u00DE\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\u00B4\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00F6\u00F2\u00F3\u00F5"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u00D6\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1149 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u00E7\u00F1\u00DE\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u00C6\u0024\u002A\u0029\u003B\u00D6"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u00C7\u00D1\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u00F0\u003A\u0023\u00D0\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0060\u00FD\u007B\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u007D\u00B8\u005D\u20AC"
          +
          "\u00B5\u00F6\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u0040\u00DD\u005B\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\\\u00D7"
          +
          "\u00FE\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u007E\u00F2\u00F3\u00F5"
          +
          "\u00E6\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\u00B4\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u005E\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1154 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0452\u0453\u0451\u0454\u0455\u0456\u0457\u0458\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u0459\u045A\u045B\u045C\u045E\u045F\u042A\u2116\u0402\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u0403\u0401\u0404\u0405\u0406\u0407\u0408\u0409\u007C\u002C\u0025\u005F\u003E\u003F"
          +
          "\u040A\u040B\u040C\u00AD\u040E\u040F\u044E\u0430\u0431\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u0446\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u0434\u0435\u0444\u0433\u0445\u0438"
          +
          "\u0439\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u043A\u043B\u043C\u043D\u043E\u043F"
          +
          "\u044F\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u0440\u0441\u0442\u0443\u0436\u0432"
          +
          "\u044C\u044B\u0437\u0448\u044D\u0449\u0447\u044A\u042E\u0410\u0411\u0426\u0414\u0415\u0424\u0413"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u0425\u0418\u0419\u041A\u041B\u041C"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u041D\u041E\u041F\u042F\u0420\u0421"
          +
          "\\\u20AC\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u0422\u0423\u0416\u0412\u042C\u042B"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u0417\u0428\u042D\u0429\u0427\u009F";

  private static final String SCP1155 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u007B\u00F1\u00C7\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u011E\u0130\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u005B\u00D1\u015F\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u0131\u003A\u00D6\u015E\'\u003D\u00DC"
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u007D\u0060\u00A6\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u20AC"
          +
          "\u00B5\u00F6\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u005D\u0024\u0040\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\u00B4\u00D7"
          +
          "\u00E7\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u007E\u00F2\u00F3\u00F5"
          +
          "\u011F\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\\\u00F9\u00FA\u00FF"
          +
          "\u00FC\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u0023\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\"\u00D9\u00DA\u009F";

  private static final String SCP1156 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0161\u00E4\u0105\u012F\u016B\u00E5\u0113\u017E\u00A2\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\u00E9\u0119\u0117\u010D\u0173\u201E\u201C\u0123\u00DF\u0021\u0024\u002A\u0029\u003B\u00AC"
          +
          "\u002D\u002F\u0160\u00C4\u0104\u012E\u016A\u00C5\u0112\u017D\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u0118\u0116\u010C\u0172\u012A\u013B\u0122\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0101\u017C\u0144\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u0156\u0157\u00E6\u0137\u00C6\u20AC"
          +
          "\u00B5\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u201D\u017A\u0100\u017B\u0143\u00AE"
          +
          "\u005E\u00A3\u012B\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u005B\u005D\u0179\u0136\u013C\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u014D\u00F6\u0146\u00F3\u00F5"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u0107\u00FC\u0142\u015B\u2019"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u014C\u00D6\u0145\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u0106\u00DC\u0141\u015A\u009F";

  private static final String SCP1157 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u007B\u00E0\u00E1\u00E3\u007D\u00E7\u00F1\u00A7\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u0060\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u20AC\u00C5\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u0023\u00C0\u00C1\u00C3\u0024\u00C7\u00D1\u00F6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\\\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u00E9\u003A\u00C4\u00D6\'\u003D\"" +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0161\u00FD\u017E\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u00B8\u00C6\u005D"
          +
          "\u00B5\u00FC\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u0160\u00DD\u017D\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u005B\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\u00B4\u00D7"
          +
          "\u00E4\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00A6\u00F2\u00F3\u00F5"
          +
          "\u00E5\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u007E\u00F9\u00FA\u00FF"
          +
          "\u00C9\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u0040\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP1158 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0452\u0491\u0451\u0454\u0455\u0456\u0457\u0458\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u0459\u045A\u045B\u045C\u045E\u045F\u042A\u2116\u0402\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u0490\u0401\u0404\u0405\u0406\u0407\u0408\u0409\u007C\u002C\u0025\u005F\u003E\u003F"
          +
          "\u040A\u040B\u040C\u00AD\u040E\u040F\u044E\u0430\u0431\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u0446\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u0434\u0435\u0444\u0433\u0445\u0438"
          +
          "\u0439\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u043A\u043B\u043C\u043D\u043E\u043F"
          +
          "\u044F\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u0440\u0441\u0442\u0443\u0436\u0432"
          +
          "\u044C\u044B\u0437\u0448\u044D\u0449\u0447\u044A\u042E\u0410\u0411\u0426\u0414\u0415\u0424\u0413"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u0425\u0418\u0419\u041A\u041B\u041C"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u041D\u041E\u041F\u042F\u0420\u0421"
          +
          "\\\u20AC\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u0422\u0423\u0416\u0412\u042C\u042B"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u0417\u0428\u042D\u0429\u0427\u009F";

  private static final String SCP1164 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u0103\u00E5\u00E7\u00F1\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u0303\u00DF\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u0102\u00C5\u00C7\u00D1\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u20AB\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0111\u0309\u0300\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u00E6\u0152\u00C6\u20AC"
          +
          "\u00B5\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u0110\u0323\u0301\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u0153\u0178\u00D7"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u00F6\u01B0\u00F3\u01A1"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\\\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u00D6\u01AF\u00D3\u01A0"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP420 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u0651\uFE7D\u0640\u200B\u0621\u0622\uFE82\u0623\u00A2\u002E\u003C\u0028\u002B\u007C"
          +
          "\u0026\uFE84\u0624\u001A\u001A\u0626\u0627\uFE8E\u0628\uFE91\u0021\u0024\u002A\u0029\u003B\u00AC"
          +
          "\u002D\u002F\u0629\u062A\uFE97\u062B\uFE9B\u062C\uFE9F\u062D\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\uFEA3\u062E\uFEA7\u062F\u0630\u0631\u0632\u0633\uFEB3\u060C\u003A\u0023\u0040\'\u003D\""
          +
          "\u0634\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\uFEB7\u0635\uFEBB\u0636\uFEBF\u0637"
          +
          "\u0638\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u0639\uFECA\uFECB\uFECC\u063A\uFECE"
          +
          "\uFECF\u00F7\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\uFED0\u0641\uFED3\u0642\uFED7\u0643"
          +
          "\uFEDB\u0644\uFEF5\uFEF6\uFEF7\uFEF8\u001A\u001A\uFEFB\uFEFC\uFEDF\u0645\uFEE3\u0646\uFEE7\u0647"
          +
          "\u061B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\uFEEB\u001A\uFEEC\u001A\u0648"
          +
          "\u061F\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u0649\uFEF0\u064A\uFEF2\uFEF3\u0660"
          +
          "\u00D7\u2007\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u0661\u0662\u001A\u0663\u0664\u0665"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u20AC\u0666\u0667\u0668\u0669\u009F";

  private static final String SCP871 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u00A0\u00E2\u00E4\u00E0\u00E1\u00E3\u00E5\u00E7\u00F1\u00DE\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u00E9\u00EA\u00EB\u00E8\u00ED\u00EE\u00EF\u00EC\u00DF\u00C6\u0024\u002A\u0029\u003B\u00D6"
          +
          "\u002D\u002F\u00C2\u00C4\u00C0\u00C1\u00C3\u00C5\u00C7\u00D1\u00A6\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00F8\u00C9\u00CA\u00CB\u00C8\u00CD\u00CE\u00CF\u00CC\u00F0\u003A\u0023\u00D0\'\u003D\""
          +
          "\u00D8\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u00AB\u00BB\u0060\u00FD\u007B\u00B1"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u00AA\u00BA\u007D\u00B8\u005D\u00A4"
          +
          "\u00B5\u00F6\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u00A1\u00BF\u0040\u00DD\u005B\u00AE"
          +
          "\u00A2\u00A3\u00A5\u00B7\u00A9\u00A7\u00B6\u00BC\u00BD\u00BE\u00AC\u007C\u00AF\u00A8\\\u00D7"
          +
          "\u00FE\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u00F4\u007E\u00F2\u00F3\u00F5"
          +
          "\u00E6\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B9\u00FB\u00FC\u00F9\u00FA\u00FF"
          +
          "\u00B4\u00F7\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00D4\u005E\u00D2\u00D3\u00D5"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00DB\u00DC\u00D9\u00DA\u009F";

  private static final String SCP875 =
      "\u0000\u0001\u0002\u0003\u009C\t\u0086\u007F\u0097\u008D\u008E\u000B\f\r\u000E\u000F" +
          "\u0010\u0011\u0012\u0013\u009D\u0085\b\u0087\u0018\u0019\u0092\u008F\u001C\u001D\u001E\u001F"
          +
          "\u0080\u0081\u0082\u0083\u0084\n\u0017\u001B\u0088\u0089\u008A\u008B\u008C\u0005\u0006\u0007"
          +
          "\u0090\u0091\u0016\u0093\u0094\u0095\u0096\u0004\u0098\u0099\u009A\u009B\u0014\u0015\u009E\u001A"
          +
          "\u0020\u0391\u0392\u0393\u0394\u0395\u0396\u0397\u0398\u0399\u005B\u002E\u003C\u0028\u002B\u0021"
          +
          "\u0026\u039A\u039B\u039C\u039D\u039E\u039F\u03A0\u03A1\u03A3\u005D\u0024\u002A\u0029\u003B\u005E"
          +
          "\u002D\u002F\u03A4\u03A5\u03A6\u03A7\u03A8\u03A9\u03AA\u03AB\u007C\u002C\u0025\u005F\u003E\u003F"
          +
          "\u00A8\u0386\u0388\u0389\u00A0\u038A\u038C\u038E\u038F\u0060\u003A\u0023\u0040\'\u003D\""
          +
          "\u0385\u0061\u0062\u0063\u0064\u0065\u0066\u0067\u0068\u0069\u03B1\u03B2\u03B3\u03B4\u03B5\u03B6"
          +
          "\u00B0\u006A\u006B\u006C\u006D\u006E\u006F\u0070\u0071\u0072\u03B7\u03B8\u03B9\u03BA\u03BB\u03BC"
          +
          "\u00B4\u007E\u0073\u0074\u0075\u0076\u0077\u0078\u0079\u007A\u03BD\u03BE\u03BF\u03C0\u03C1\u03C3"
          +
          "\u00A3\u03AC\u03AD\u03AE\u03CA\u03AF\u03CC\u03CD\u03CB\u03CE\u03C2\u03C4\u03C5\u03C6\u03C7\u03C8"
          +
          "\u007B\u0041\u0042\u0043\u0044\u0045\u0046\u0047\u0048\u0049\u00AD\u03C9\u0390\u03B0\u2018\u2015"
          +
          "\u007D\u004A\u004B\u004C\u004D\u004E\u004F\u0050\u0051\u0052\u00B1\u00BD\u001A\u0387\u2019\u00A6"
          +
          "\\\u001A\u0053\u0054\u0055\u0056\u0057\u0058\u0059\u005A\u00B2\u00A7\u001A\u001A\u00AB\u00AC"
          +
          "\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039\u00B3\u00A9\u001A\u001A\u00BB\u009F";

  public static final XIEbcdicNTranslator TRANSLATOR_CP1025 =
      new XIEbcdicNTranslator(SCP2CP(SCP1025));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1026 =
      new XIEbcdicNTranslator(SCP2CP(SCP1026));

  public static final XIEbcdicNTranslator TRANSLATOR_CP1112 =
      new XIEbcdicNTranslator(SCP2CP(SCP1112));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1122 =
      new XIEbcdicNTranslator(SCP2CP(SCP1122));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1123 =
      new XIEbcdicNTranslator(SCP2CP(SCP1123));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1125 =
      new XIEbcdicNTranslator(SCP2CP(SCP1125));

  public static final XIEbcdicNTranslator TRANSLATOR_CP1130 =
      new XIEbcdicNTranslator(SCP2CP(SCP1130));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1132 =
      new XIEbcdicNTranslator(SCP2CP(SCP1132));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1137 =
      new XIEbcdicNTranslator(SCP2CP(SCP1137));

  public static final XIEbcdicNTranslator TRANSLATOR_CP1142 =
      new XIEbcdicNTranslator(SCP2CP(SCP1142));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1145 =
      new XIEbcdicNTranslator(SCP2CP(SCP1145));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1146 =
      new XIEbcdicNTranslator(SCP2CP(SCP1146));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1148 =
      new XIEbcdicNTranslator(SCP2CP(SCP1148));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1149 =
      new XIEbcdicNTranslator(SCP2CP(SCP1149));

  public static final XIEbcdicNTranslator TRANSLATOR_CP1154 =
      new XIEbcdicNTranslator(SCP2CP(SCP1154));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1155 =
      new XIEbcdicNTranslator(SCP2CP(SCP1155));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1156 =
      new XIEbcdicNTranslator(SCP2CP(SCP1156));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1157 =
      new XIEbcdicNTranslator(SCP2CP(SCP1157));
  public static final XIEbcdicNTranslator TRANSLATOR_CP1158 =
      new XIEbcdicNTranslator(SCP2CP(SCP1158));

  public static final XIEbcdicNTranslator TRANSLATOR_CP1164 =
      new XIEbcdicNTranslator(SCP2CP(SCP1164));

  public static final XIEbcdicNTranslator TRANSLATOR_CP420 =
      new XIEbcdicNTranslator(SCP2CP(SCP420));
  public static final XIEbcdicNTranslator TRANSLATOR_CP871 =
      new XIEbcdicNTranslator(SCP2CP(SCP871));
  public static final XIEbcdicNTranslator TRANSLATOR_CP875 =
      new XIEbcdicNTranslator(SCP2CP(SCP875));

}
