/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.iceberg;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.netflix.iceberg.PartitionField;
import com.netflix.iceberg.Schema;
import com.netflix.iceberg.StructLike;
import com.netflix.iceberg.exceptions.ValidationException;
import com.netflix.iceberg.transforms.Transforms;
import com.netflix.iceberg.types.Type;
import com.netflix.iceberg.types.Types;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

public class PartitionSpec
implements Serializable {
    private static final int PARTITION_DATA_ID_START = 1000;
    private final Schema schema;
    private final int specId;
    private final PartitionField[] fields;
    private transient Map<Integer, PartitionField> fieldsBySourceId = null;
    private transient Map<String, PartitionField> fieldsByName = null;
    private transient Class<?>[] javaClasses = null;
    private transient List<PartitionField> fieldList = null;
    private static final PartitionSpec UNPARTITIONED_SPEC = new PartitionSpec(new Schema(new Types.NestedField[0]), 0, (List<PartitionField>)ImmutableList.of());

    private PartitionSpec(Schema schema, int specId, List<PartitionField> fields) {
        this.schema = schema;
        this.specId = specId;
        this.fields = new PartitionField[fields.size()];
        for (int i = 0; i < this.fields.length; ++i) {
            this.fields[i] = fields.get(i);
        }
    }

    public Schema schema() {
        return this.schema;
    }

    public int specId() {
        return this.specId;
    }

    public List<PartitionField> fields() {
        return this.lazyFieldList();
    }

    public PartitionField getFieldBySourceId(int fieldId) {
        return this.lazyFieldsBySourceId().get(fieldId);
    }

    public Types.StructType partitionType() {
        ArrayList structFields = Lists.newArrayListWithExpectedSize((int)this.fields.length);
        for (int i = 0; i < this.fields.length; ++i) {
            PartitionField field = this.fields[i];
            Type sourceType = this.schema.findType(field.sourceId());
            Type resultType = field.transform().getResultType(sourceType);
            structFields.add(Types.NestedField.optional(1000 + i, field.name(), resultType));
        }
        return Types.StructType.of(structFields);
    }

    public Class<?>[] javaClasses() {
        if (this.javaClasses == null) {
            this.javaClasses = new Class[this.fields.length];
            for (int i = 0; i < this.fields.length; ++i) {
                PartitionField field = this.fields[i];
                Type sourceType = this.schema.findType(field.sourceId());
                Type result = field.transform().getResultType(sourceType);
                this.javaClasses[i] = result.typeId().javaClass();
            }
        }
        return this.javaClasses;
    }

    private <T> T get(StructLike data, int pos, Class<?> javaClass) {
        return (T)data.get(pos, javaClass);
    }

    private String escape(String string) {
        try {
            return URLEncoder.encode(string, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    public String partitionToPath(StructLike data) {
        StringBuilder sb = new StringBuilder();
        Class<?>[] javaClasses = this.javaClasses();
        for (int i = 0; i < javaClasses.length; ++i) {
            PartitionField field = this.fields[i];
            String valueString = field.transform().toHumanString(this.get(data, i, javaClasses[i]));
            if (i > 0) {
                sb.append("/");
            }
            sb.append(field.name()).append("=").append(this.escape(valueString));
        }
        return sb.toString();
    }

    public boolean compatibleWith(PartitionSpec other) {
        if (this.equals(other)) {
            return true;
        }
        if (this.fields.length != other.fields.length) {
            return false;
        }
        for (int i = 0; i < this.fields.length; ++i) {
            PartitionField thisField = this.fields[i];
            PartitionField thatField = other.fields[i];
            if (thisField.sourceId() == thatField.sourceId() && thisField.transform().toString().equals(thatField.transform().toString())) continue;
            return false;
        }
        return true;
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other == null || this.getClass() != other.getClass()) {
            return false;
        }
        PartitionSpec that = (PartitionSpec)other;
        if (this.specId != that.specId) {
            return false;
        }
        return Arrays.equals(this.fields, that.fields);
    }

    public int hashCode() {
        return Objects.hashCode(Arrays.hashCode(this.fields));
    }

    private List<PartitionField> lazyFieldList() {
        if (this.fieldList == null) {
            this.fieldList = ImmutableList.copyOf((Object[])this.fields);
        }
        return this.fieldList;
    }

    private Map<String, PartitionField> lazyFieldsByName() {
        if (this.fieldsByName == null) {
            ImmutableMap.Builder builder = ImmutableMap.builder();
            for (PartitionField field : this.fields) {
                builder.put((Object)field.name(), (Object)field);
            }
            this.fieldsByName = builder.build();
        }
        return this.fieldsByName;
    }

    private Map<Integer, PartitionField> lazyFieldsBySourceId() {
        if (this.fieldsBySourceId == null) {
            ImmutableMap.Builder byIdBuilder = ImmutableMap.builder();
            for (PartitionField field : this.fields) {
                byIdBuilder.put((Object)field.sourceId(), (Object)field);
            }
            this.fieldsBySourceId = byIdBuilder.build();
        }
        return this.fieldsBySourceId;
    }

    public Set<Integer> identitySourceIds() {
        HashSet sourceIds = Sets.newHashSet();
        List<PartitionField> fields = this.fields();
        for (PartitionField field : fields) {
            if (!"identity".equals(field.transform().toString())) continue;
            sourceIds.add(field.sourceId());
        }
        return sourceIds;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (PartitionField field : this.fields) {
            sb.append("\n");
            sb.append("  ").append(field.name()).append(": ").append(field.transform()).append("(").append(field.sourceId()).append(")");
        }
        if (this.fields.length > 0) {
            sb.append("\n");
        }
        sb.append("]");
        return sb.toString();
    }

    public static PartitionSpec unpartitioned() {
        return UNPARTITIONED_SPEC;
    }

    public static Builder builderFor(Schema schema) {
        return new Builder(schema);
    }

    public static void checkCompatibility(PartitionSpec spec, Schema schema) {
        for (PartitionField field : spec.fields) {
            Type sourceType = schema.findType(field.sourceId());
            ValidationException.check(sourceType.isPrimitiveType(), "Cannot partition by non-primitive source field: %s", sourceType);
            ValidationException.check(field.transform().canTransform(sourceType), "Invalid source type %s for transform: %s", sourceType, field.transform());
        }
    }

    public static class Builder {
        private final Schema schema;
        private final List<PartitionField> fields = Lists.newArrayList();
        private final Set<String> partitionNames = Sets.newHashSet();
        private int specId = 0;

        private Builder(Schema schema) {
            this.schema = schema;
        }

        private void checkAndAddPartitionName(String name) {
            Preconditions.checkArgument((name != null && !name.isEmpty() ? 1 : 0) != 0, (String)"Cannot use empty or null partition name: %s", (Object[])new Object[]{name});
            Preconditions.checkArgument((!this.partitionNames.contains(name) ? 1 : 0) != 0, (String)"Cannot use partition name more than once: %s", (Object[])new Object[]{name});
            this.partitionNames.add(name);
        }

        public Builder withSpecId(int specId) {
            this.specId = specId;
            return this;
        }

        private Types.NestedField findSourceColumn(String sourceName) {
            Types.NestedField sourceColumn = this.schema.findField(sourceName);
            Preconditions.checkNotNull((Object)sourceColumn, (String)"Cannot find source column: %s", (Object[])new Object[]{sourceName});
            return sourceColumn;
        }

        public Builder identity(String sourceName) {
            this.checkAndAddPartitionName(sourceName);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), sourceName, Transforms.identity(sourceColumn.type())));
            return this;
        }

        public Builder year(String sourceName) {
            String name = sourceName + "_year";
            this.checkAndAddPartitionName(name);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), name, Transforms.year(sourceColumn.type())));
            return this;
        }

        public Builder month(String sourceName) {
            String name = sourceName + "_month";
            this.checkAndAddPartitionName(name);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), name, Transforms.month(sourceColumn.type())));
            return this;
        }

        public Builder day(String sourceName) {
            String name = sourceName + "_day";
            this.checkAndAddPartitionName(name);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), name, Transforms.day(sourceColumn.type())));
            return this;
        }

        public Builder hour(String sourceName) {
            String name = sourceName + "_hour";
            this.checkAndAddPartitionName(name);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), name, Transforms.hour(sourceColumn.type())));
            return this;
        }

        public Builder bucket(String sourceName, int numBuckets) {
            String name = sourceName + "_bucket";
            this.checkAndAddPartitionName(name);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), name, Transforms.bucket(sourceColumn.type(), numBuckets)));
            return this;
        }

        public Builder truncate(String sourceName, int width) {
            String name = sourceName + "_trunc";
            this.checkAndAddPartitionName(name);
            Types.NestedField sourceColumn = this.findSourceColumn(sourceName);
            this.fields.add(new PartitionField(sourceColumn.fieldId(), name, Transforms.truncate(sourceColumn.type(), width)));
            return this;
        }

        public Builder add(int sourceId, String name, String transform) {
            this.checkAndAddPartitionName(name);
            Types.NestedField column = this.schema.findField(sourceId);
            Preconditions.checkNotNull((Object)column, (String)"Cannot find source column: %d", (Object[])new Object[]{sourceId});
            this.fields.add(new PartitionField(sourceId, name, Transforms.fromString(column.type(), transform)));
            return this;
        }

        public PartitionSpec build() {
            PartitionSpec spec = new PartitionSpec(this.schema, this.specId, this.fields);
            PartitionSpec.checkCompatibility(spec, this.schema);
            return spec;
        }
    }
}

