/*
 * Decompiled with CFR 0.152.
 */
package io.github.muntashirakon.adb;

import android.annotation.SuppressLint;
import android.os.Build;
import android.util.Log;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import io.github.muntashirakon.adb.AndroidPubkey;
import io.github.muntashirakon.adb.KeyPair;
import io.github.muntashirakon.adb.PairingAuthCtx;
import io.github.muntashirakon.adb.SslUtils;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.InvalidKeyException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.interfaces.RSAPublicKey;
import java.util.Arrays;
import java.util.Objects;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLSocket;

@RequiresApi(value=9)
public final class PairingConnectionCtx
implements Closeable {
    public static final String TAG = PairingConnectionCtx.class.getSimpleName();
    public static final String EXPORTED_KEY_LABEL = "adb-label\u0000";
    public static final int EXPORT_KEY_SIZE = 64;
    private final String mHost;
    private final int mPort;
    private final byte[] mPswd;
    private final PeerInfo mPeerInfo;
    private final SSLContext mSslContext;
    private final Role mRole = Role.Client;
    private DataInputStream mInputStream;
    private DataOutputStream mOutputStream;
    private PairingAuthCtx mPairingAuthCtx;
    private State mState = State.Ready;

    public PairingConnectionCtx(@NonNull String host, int port, @NonNull byte[] pswd, @NonNull KeyPair keyPair, @NonNull String deviceName) throws NoSuchAlgorithmException, KeyManagementException, InvalidKeyException {
        this.mHost = Objects.requireNonNull(host);
        this.mPort = port;
        this.mPswd = Objects.requireNonNull(pswd);
        this.mPeerInfo = new PeerInfo(0, AndroidPubkey.encodeWithName((RSAPublicKey)keyPair.getPublicKey(), Objects.requireNonNull(deviceName)));
        this.mSslContext = SslUtils.getSslContext(keyPair);
    }

    public PairingConnectionCtx(@NonNull String host, int port, @NonNull byte[] pswd, @NonNull PrivateKey privateKey, @NonNull Certificate certificate, @NonNull String deviceName) throws NoSuchAlgorithmException, KeyManagementException, InvalidKeyException {
        this(host, port, pswd, new KeyPair(Objects.requireNonNull(privateKey), Objects.requireNonNull(certificate)), deviceName);
    }

    public void start() throws IOException {
        if (this.mState != State.Ready) {
            throw new IOException("Connection is not ready yet.");
        }
        this.mState = State.ExchangingMsgs;
        this.setupTlsConnection();
        while (true) {
            switch (this.mState) {
                case ExchangingMsgs: {
                    if (!this.doExchangeMsgs()) {
                        this.notifyResult();
                        throw new IOException("Exchanging message wasn't successful.");
                    }
                    this.mState = State.ExchangingPeerInfo;
                    break;
                }
                case ExchangingPeerInfo: {
                    if (!this.doExchangePeerInfo()) {
                        this.notifyResult();
                        throw new IOException("Could not exchange peer info.");
                    }
                    this.notifyResult();
                    return;
                }
                case Ready: 
                case Stopped: {
                    throw new IOException("Connection closed with errors.");
                }
            }
        }
    }

    private void notifyResult() {
        this.mState = State.Stopped;
    }

    private void setupTlsConnection() throws IOException {
        Socket socket;
        if (this.mRole == Role.Server) {
            SSLServerSocket sslServerSocket = (SSLServerSocket)this.mSslContext.getServerSocketFactory().createServerSocket(this.mPort);
            socket = sslServerSocket.accept();
        } else {
            socket = new Socket(this.mHost, this.mPort);
        }
        socket.setTcpNoDelay(true);
        SSLSocket sslSocket = (SSLSocket)this.mSslContext.getSocketFactory().createSocket(socket, this.mHost, this.mPort, true);
        sslSocket.startHandshake();
        Log.d((String)TAG, (String)"Handshake succeeded.");
        this.mInputStream = new DataInputStream(sslSocket.getInputStream());
        this.mOutputStream = new DataOutputStream(sslSocket.getOutputStream());
        byte[] keyMaterial = this.exportKeyingMaterial(sslSocket, 64);
        byte[] passwordBytes = new byte[this.mPswd.length + keyMaterial.length];
        System.arraycopy(this.mPswd, 0, passwordBytes, 0, this.mPswd.length);
        System.arraycopy(keyMaterial, 0, passwordBytes, this.mPswd.length, keyMaterial.length);
        PairingAuthCtx pairingAuthCtx = PairingAuthCtx.createAlice(passwordBytes);
        if (pairingAuthCtx == null) {
            throw new IOException("Unable to create PairingAuthCtx.");
        }
        this.mPairingAuthCtx = pairingAuthCtx;
    }

    @SuppressLint(value={"PrivateApi"})
    private byte[] exportKeyingMaterial(SSLSocket sslSocket, int length) throws SSLException {
        try {
            Class<?> conscryptClass;
            if (SslUtils.isCustomConscrypt()) {
                conscryptClass = Class.forName("org.conscrypt.Conscrypt");
            } else {
                if (Build.VERSION.SDK_INT < 29) {
                    throw new SSLException("TLSv1.3 isn't supported on your platform. Use custom Conscrypt library instead.");
                }
                conscryptClass = Class.forName("com.android.org.conscrypt.Conscrypt");
            }
            Method exportKeyingMaterial = conscryptClass.getMethod("exportKeyingMaterial", SSLSocket.class, String.class, byte[].class, Integer.TYPE);
            return (byte[])exportKeyingMaterial.invoke(null, sslSocket, EXPORTED_KEY_LABEL, null, length);
        }
        catch (SSLException e) {
            throw e;
        }
        catch (Throwable th) {
            throw new SSLException(th);
        }
    }

    private void writeHeader(@NonNull PairingPacketHeader header, @NonNull byte[] payload) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(6).order(ByteOrder.BIG_ENDIAN);
        header.writeTo(buffer);
        this.mOutputStream.write(buffer.array());
        this.mOutputStream.write(payload);
    }

    @Nullable
    private PairingPacketHeader readHeader() throws IOException {
        byte[] bytes = new byte[6];
        this.mInputStream.readFully(bytes);
        ByteBuffer buffer = ByteBuffer.wrap(bytes).order(ByteOrder.BIG_ENDIAN);
        return PairingPacketHeader.readFrom(buffer);
    }

    @NonNull
    private PairingPacketHeader createHeader(byte type, int payloadSize) {
        return new PairingPacketHeader(1, type, payloadSize);
    }

    private boolean checkHeaderType(byte expected, byte actual) {
        if (expected != actual) {
            Log.e((String)TAG, (String)("Unexpected header type (expected=" + expected + " actual=" + actual + ")"));
            return false;
        }
        return true;
    }

    private boolean doExchangeMsgs() throws IOException {
        byte[] msg = this.mPairingAuthCtx.getMsg();
        PairingPacketHeader ourHeader = this.createHeader((byte)0, msg.length);
        this.writeHeader(ourHeader, msg);
        PairingPacketHeader theirHeader = this.readHeader();
        if (theirHeader == null || !this.checkHeaderType((byte)0, theirHeader.type)) {
            return false;
        }
        byte[] theirMsg = new byte[theirHeader.payloadSize];
        this.mInputStream.readFully(theirMsg);
        try {
            return this.mPairingAuthCtx.initCipher(theirMsg);
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Unable to initialize pairing cipher");
            throw (IOException)new IOException().initCause(e);
        }
    }

    private boolean doExchangePeerInfo() throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(8192).order(ByteOrder.BIG_ENDIAN);
        this.mPeerInfo.writeTo(buffer);
        byte[] outBuffer = this.mPairingAuthCtx.encrypt(buffer.array());
        if (outBuffer == null) {
            Log.e((String)TAG, (String)"Failed to encrypt peer info");
            return false;
        }
        PairingPacketHeader ourHeader = this.createHeader((byte)1, outBuffer.length);
        this.writeHeader(ourHeader, outBuffer);
        PairingPacketHeader theirHeader = this.readHeader();
        if (theirHeader == null || !this.checkHeaderType((byte)1, theirHeader.type)) {
            return false;
        }
        byte[] theirMsg = new byte[theirHeader.payloadSize];
        this.mInputStream.readFully(theirMsg);
        byte[] decryptedMsg = this.mPairingAuthCtx.decrypt(theirMsg);
        if (decryptedMsg == null) {
            Log.e((String)TAG, (String)"Unsupported payload while decrypting peer info.");
            return false;
        }
        if (decryptedMsg.length != 8192) {
            Log.e((String)TAG, (String)("Got size=" + decryptedMsg.length + " PeerInfo.size=" + 8192));
            return false;
        }
        PeerInfo theirPeerInfo = PeerInfo.readFrom(ByteBuffer.wrap(decryptedMsg));
        Log.d((String)TAG, (String)theirPeerInfo.toString());
        return true;
    }

    @Override
    public void close() {
        Arrays.fill(this.mPswd, (byte)0);
        try {
            this.mInputStream.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            this.mOutputStream.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (this.mState != State.Ready) {
            this.mPairingAuthCtx.destroy();
        }
    }

    private static class PairingPacketHeader {
        public static final byte CURRENT_KEY_HEADER_VERSION = 1;
        public static final byte MIN_SUPPORTED_KEY_HEADER_VERSION = 1;
        public static final byte MAX_SUPPORTED_KEY_HEADER_VERSION = 1;
        public static final int MAX_PAYLOAD_SIZE = 16384;
        public static final byte PAIRING_PACKET_HEADER_SIZE = 6;
        public static final byte SPAKE2_MSG = 0;
        public static final byte PEER_INFO = 1;
        private final byte version;
        private final byte type;
        private final int payloadSize;

        @Nullable
        public static PairingPacketHeader readFrom(@NonNull ByteBuffer buffer) {
            byte version = buffer.get();
            byte type = buffer.get();
            int payload = buffer.getInt();
            if (version < 1 || version > 1) {
                Log.e((String)TAG, (String)("PairingPacketHeader version mismatch (us=1 them=" + version + ")"));
                return null;
            }
            if (type != 0 && type != 1) {
                Log.e((String)TAG, (String)("Unknown PairingPacket type " + type));
                return null;
            }
            if (payload <= 0 || payload > 16384) {
                Log.e((String)TAG, (String)("Header payload not within a safe payload size (size=" + payload + ")"));
                return null;
            }
            return new PairingPacketHeader(version, type, payload);
        }

        public PairingPacketHeader(byte version, byte type, int payloadSize) {
            this.version = version;
            this.type = type;
            this.payloadSize = payloadSize;
        }

        public void writeTo(@NonNull ByteBuffer buffer) {
            buffer.put(this.version).put(this.type).putInt(this.payloadSize);
        }

        @NonNull
        public String toString() {
            return "PairingPacketHeader{version=" + this.version + ", type=" + this.type + ", payloadSize=" + this.payloadSize + '}';
        }
    }

    private static class PeerInfo {
        public static final int MAX_PEER_INFO_SIZE = 8192;
        public static final byte ADB_RSA_PUB_KEY = 0;
        public static final byte ADB_DEVICE_GUID = 0;
        private final byte type;
        private final byte[] data = new byte[8191];

        @NonNull
        public static PeerInfo readFrom(@NonNull ByteBuffer buffer) {
            byte type = buffer.get();
            byte[] data = new byte[8191];
            buffer.get(data);
            return new PeerInfo(type, data);
        }

        public PeerInfo(byte type, byte[] data) {
            this.type = type;
            System.arraycopy(data, 0, this.data, 0, Math.min(data.length, 8191));
        }

        public void writeTo(@NonNull ByteBuffer buffer) {
            buffer.put(this.type).put(this.data);
        }

        @NonNull
        public String toString() {
            return "PeerInfo{type=" + this.type + ", data=" + Arrays.toString(this.data) + '}';
        }
    }

    static enum Role {
        Client,
        Server;

    }

    private static enum State {
        Ready,
        ExchangingMsgs,
        ExchangingPeerInfo,
        Stopped;

    }
}

