/*
 * Decompiled with CFR 0.152.
 */
package com.github.jikoo.regionerator.util;

import com.github.jikoo.regionerator.util.ExpirationMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class BatchExpirationLoadingCache<K, V> {
    private final Map<K, V> internal = new ConcurrentHashMap();
    private final Collection<K> expired = Collections.synchronizedSet(new LinkedHashSet());
    private final AtomicBoolean expirationQueued = new AtomicBoolean();
    private final ExpirationMap<K> expirationMap;
    private final Function<K, V> load;
    private final Consumer<Collection<V>> expirationConsumer;
    private final int maxBatchSize;
    private final long batchDelay;

    public BatchExpirationLoadingCache(long retention, @NotNull Function<K, V> load, @NotNull Consumer<Collection<V>> expirationConsumer) {
        this(retention, load, expirationConsumer, 1024, 5000L);
    }

    public BatchExpirationLoadingCache(long retention, @NotNull Function<K, V> load, @NotNull Consumer<Collection<V>> expirationConsumer, int maxBatchSize, long batchDelay) {
        this.expirationMap = new ExpirationMap(retention);
        this.load = key -> {
            Object value = load.apply(key);
            if (value != null) {
                this.expirationMap.add(key);
            }
            this.checkExpiration();
            return value;
        };
        this.expirationConsumer = expirationConsumer;
        if (maxBatchSize < 1) {
            throw new IllegalArgumentException("Max batch size cannot be smaller than 1");
        }
        this.maxBatchSize = maxBatchSize;
        this.batchDelay = batchDelay;
    }

    @NotNull
    public CompletableFuture<V> get(@NotNull K key) {
        V value = this.getIfPresent(key);
        if (value != null) {
            return CompletableFuture.completedFuture(value);
        }
        return CompletableFuture.supplyAsync(() -> this.load.apply(key));
    }

    @Nullable
    public V getIfPresent(@NotNull K key) {
        V value = this.internal.get(key);
        if (value != null) {
            this.expirationMap.add(key);
        }
        this.checkExpiration();
        return value;
    }

    @NotNull
    public V computeIfAbsent(@NotNull K key, Function<K, V> supplier) {
        V value = this.getIfPresent(key);
        if (value != null) {
            return value;
        }
        value = supplier.apply(key);
        this.put(key, value);
        return value;
    }

    public void put(@NotNull K key, @NotNull V value) {
        this.internal.put(key, value);
        this.expirationMap.add(key);
    }

    public void remove(@NotNull K key) {
        if (this.internal.remove(key) != null) {
            this.expirationMap.remove(key);
            this.expired.remove(key);
        }
    }

    private void checkExpiration() {
        if (!this.expirationQueued.get()) {
            this.expired.addAll(this.expirationMap.doExpiration());
        }
        if (this.expired.isEmpty() || !this.expirationQueued.compareAndSet(false, true)) {
            return;
        }
        new Thread(() -> {
            if (this.expired.size() < this.maxBatchSize) {
                try {
                    Thread.sleep(this.batchDelay);
                    this.expired.addAll(this.expirationMap.doExpiration());
                }
                catch (InterruptedException e) {
                    System.err.println("Encountered exception while attempting to await larger batch:");
                    e.printStackTrace();
                }
            }
            ArrayList<V> expiredValues = new ArrayList<V>();
            Iterator<K> expiredIterator = this.expired.iterator();
            while (expiredValues.size() < this.maxBatchSize && expiredIterator.hasNext()) {
                K expiredKey = expiredIterator.next();
                expiredIterator.remove();
                V value = this.expirationMap.contains(expiredKey) ? this.internal.get(expiredKey) : this.internal.remove(expiredKey);
                if (value == null) continue;
                expiredValues.add(value);
            }
            this.expirationConsumer.accept(expiredValues);
            this.expirationQueued.set(false);
            this.checkExpiration();
        }, "BatchExpiration").start();
    }

    public void lazyExpireAll() {
        this.expired.addAll(this.internal.keySet());
        this.checkExpiration();
    }

    public void expireAll() {
        this.expired.clear();
        if (this.internal.size() <= this.maxBatchSize) {
            this.expirationConsumer.accept(this.internal.values());
            this.internal.clear();
        } else {
            Iterator<V> iterator = this.internal.values().iterator();
            ArrayList subsets = new ArrayList();
            while (iterator.hasNext()) {
                ArrayList<V> subset = new ArrayList<V>(this.maxBatchSize);
                for (int i = 0; i < this.maxBatchSize && iterator.hasNext(); ++i) {
                    subset.add(iterator.next());
                    iterator.remove();
                }
                subsets.add(subset);
            }
            for (Collection collection : subsets) {
                this.expirationConsumer.accept(collection);
            }
        }
        this.expirationMap.clear();
    }

    public int getCached() {
        return this.internal.size();
    }

    public int getQueued() {
        return this.expired.size();
    }
}

