package com.amity.socialcloud.sdk.api.social.comment.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.ekoapp.ekosdk.internal.usecase.comment.CommentPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommentQuery internal constructor(
    private val referenceId: String,
    private val referenceType: String,
    private val isFilterByParentId: Boolean,
    private val parentId: String?,
    private val isDeleted: Boolean?,
    private val sortOption: AmityCommentSortOption,
    private val dataTypeFilter: AmityCommentDataTypeFilter?
) {

    fun loader(): AmityCommentLoader {
        return AmityCommentLoader(
            referenceId,
            referenceType,
            isFilterByParentId,
            parentId,
            isDeleted,
            sortOption
        )
    }

    /**
     * Queries for comments
     * @return LiveCollection of Comment as Flowable<PagingData<AmityComment>>
     */
    fun query(): Flowable<PagingData<AmityComment>> {
        return CommentPagingDataUseCase().execute(
            referenceId,
            referenceType,
            isFilterByParentId,
            parentId,
            isDeleted,
            sortOption,
            dataTypeFilter
        )
    }

    class Builder internal constructor() {
        private lateinit var referenceId: String
        private lateinit var referenceType: String
        private var parentId: String? = null
        private var isFilterByParentId: Boolean = false
        private var isDeleted: Boolean? = null
        private var sortOption: AmityCommentSortOption = AmityCommentSortOption.LAST_CREATED
        private var dataTypeFilter: AmityCommentDataTypeFilter? = null

        /**
         * @param [includeDeleted] to specify includeDeleted filter.
         * When [includeDeleted] is true, the result will include soft-deleted comments.
         * When [includeDeleted] is false, the result will exclude soft-deleted comments.
         * The default value of [includeDeleted] is true.
         * @return the same AmityCommentQuery.Builder instance.
         */
        fun includeDeleted(includeDeleted: Boolean): Builder {
            return apply {
                this.isDeleted = if(includeDeleted) null else false
            }
        }

        /**
         * @param [parentId] a commentId.
         * When the value is null, the result contains only root-level comments.
         * When the value is not null, the result contains the replies the the specified commentId.
         * When the function is not called, the result contains comments from all levels.
         * @return the same AmityCommentQuery.Builder instance.
         */
        fun parentId(parentId: String?): Builder {
            this.parentId = parentId
            this.isFilterByParentId = true
            return this
        }

        /**
         * @param [sortOption] to specify the sorting option.
         * When [sortOption] is AmityCommentSortOption.FIRST_CREATED, posts will be sorted by createdAt property ASC.
         * When [sortOption] is AmityCommentSortOption.LAST_CREATED, posts will be sorted by createdAt property DESC.
         * The default value of [sortOption] is AmityCommentSortOption.LAST_CREATED.
         * @return the same AmityCommentQuery.Builder instance.
         */
        fun sortBy(sortOption: AmityCommentSortOption): Builder {
            return apply { this.sortOption = sortOption }
        }

        /**
         * @param [filter] to filter by dataTypes
         * Use AmityCommentDataTypeFilter.Exact to filter for exact matching of specified dataTypes.
         * Use AmityCommentDataTypeFilter.Any to filter for any matching of specified dataTypes.
         * The default value of [filter] is null resulting in no filtering.
         * @return the same AmityCommentQuery.Builder instance.
         */
        fun dataTypes(filter: AmityCommentDataTypeFilter): Builder {
            return apply {
                this.dataTypeFilter = filter
            }
        }

        internal fun referenceType(referenceType: String): Builder {
            return apply { this.referenceType = referenceType }
        }

        internal fun referenceId(referenceId: String): Builder {
            return apply { this.referenceId = referenceId }
        }

        /**
         * Instantiates AmityCommentQuery with built params.
         * @return new instance of AmityCommentQuery.
         */
        fun build(): AmityCommentQuery {
            return AmityCommentQuery(
                referenceId = referenceId,
                referenceType = referenceType,
                isFilterByParentId = isFilterByParentId,
                parentId = parentId,
                isDeleted = isDeleted,
                sortOption = sortOption,
                dataTypeFilter = dataTypeFilter
            )
        }
    }
}