package com.amity.socialcloud.sdk.social.data.story

import androidx.room.Dao
import androidx.room.Query
import com.amity.socialcloud.sdk.entity.social.story.StoryEntity
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.ekoapp.ekosdk.internal.data.dao.EkoObjectDao
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime


@Dao
abstract class StoryDao : EkoObjectDao<StoryEntity>() {

    @Query("select * from story where storyId = :storyId")
    abstract fun getStory(storyId: String): Flowable<StoryEntity>

    @Query("UPDATE story set isDeleted = 1 where storyId = :storyId")
    abstract fun softDeleteStory(storyId: String): Completable

    @Query("DELETE from story where storyId = :storyId")
    abstract fun hardDeleteStory(storyId: String): Completable

    @Query("DELETE from story where storyId = :storyId")
    abstract fun hardDeleteNow(storyId: String)

    @Query("update story set storyId = :storyId where storyId = :storyId")
    abstract fun dummyUpdateStory(storyId: String): Completable

    @Query("UPDATE story set commentCount = MAX(commentCount - 1, 0)  where storyId = :storyId")
    abstract fun decrementCommentCount(storyId: String)

    @Query("delete from story")
    abstract override fun deleteAll()

    @Query("select * from story where uniqueId = :uniqueId limit 1")
    abstract fun getByIdNowImpl(uniqueId: String): StoryEntity?

    @Query("update story set syncState = :syncState where uniqueId = :uniqueId")
    abstract fun updateSyncState(uniqueId: String, syncState: String): Completable

    override fun getByIdNow(uniqueId: String): StoryEntity? {
        return getByIdNowImpl(uniqueId)
    }

    @Query(
        "SELECT *" +
                " from story" +
                " where story.targetType = :targetType" +
                " and story.targetId = :targetId " +
                " and story.updatedAt > :now" +
                " and story.uniqueId not in " +
                "(" +
                "SELECT amity_paging_id.id" +
                " from amity_paging_id" +
                " where amity_paging_id.hash = (:hash)" +
                " and amity_paging_id.nonce = (:nonce) " +
                ")" +
                " order by story.updatedAt  desc" +
                " limit 1"
    )
    abstract fun getLatestStoryImpl(
        targetType: String,
        targetId: String,
        hash: Int,
        nonce: Int,
        now: DateTime = DateTime.now()
    ): Flowable<StoryEntity>

    fun getLatestStory(
        targetType: AmityStory.TargetType,
        targetId: String,
        hash: Int,
        nonce: Int,
    ): Flowable<StoryEntity> {
        return getLatestStoryImpl(
            targetType = targetType.apiKey,
            targetId = targetId,
            hash = hash,
            nonce = nonce
        )
    }

    fun getUniqueIdByStoryId(storyId: String): String? {
        return getUniqueIdByStoryIdImpl(storyId)
    }

    @Query(
        "SELECT story.uniqueId" +
                " from story" +
                " where story.storyId = :storyId" +
                " LIMIT 1"
    )
    abstract fun getUniqueIdByStoryIdImpl(storyId: String): String?

    @Query(
        "SELECT story.*" +
                " from story" +
                " where story.storyId = :storyId" +
                " LIMIT 1"
    )
    abstract fun observeStoryImpl(storyId: String?): Flowable<StoryEntity>

    fun observeStory(storyId: String): Flowable<StoryEntity> {
        return observeStoryImpl(storyId)
    }

    @Query(
        "SELECT story.*" +
                " from story" +
                " where story.storyId = :storyId" +
                " LIMIT 1"
    )
    abstract fun getByStoryIdNowImpl(storyId: String?): StoryEntity?

    fun getByStoryIdNow(storyId: String): StoryEntity? {
        return getByStoryIdNowImpl(storyId)
    }

    @Query(
        "SELECT storyExpiresAt" +
                " from story where targetType = :targetType" +
                " and targetId = :targetId" +
                " and syncState IN (:syncStates)" +
                " order by storyExpiresAt DESC LIMIT 1"
    )
    abstract fun getHighestStoryExpiresAt(
        targetType: String,
        targetId: String,
        syncStates: List<String>
    ): DateTime?

    @Query(
        "SELECT COUNT(*)" +
                " from story where targetType = :targetType" +
                " and targetId = :targetId" +
                " and syncState IN (:syncStates)"
    )
    abstract fun getStoryCount(
        targetType: String,
        targetId: String,
        syncStates: List<String>
    ): Int

    @Query("UPDATE story set syncState = 'failed' where syncState = 'syncing'")
    abstract fun initSyncStateOnStartup()

    @Query(
        "SELECT story.*" +
                " from story where targetType = :targetType and targetId = :targetId and syncState != 'synced'"
    )
    abstract fun findCache(targetType: String, targetId: String): List<StoryEntity>

}