package com.amity.socialcloud.sdk.social.data.story

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.social.story.AmityStorySortOption
import com.amity.socialcloud.sdk.dto.social.story.StoryQueryDto
import com.amity.socialcloud.sdk.entity.social.story.StoryEntity
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

internal class StoryMediator(
    private val targetType: AmityStory.TargetType,
    private val targetId: String,
    private val sortOption: AmityStorySortOption
) : DynamicQueryStreamMediator<StoryEntity, StoryQueryDto, AmityStory>(
    nonce = AmityNonce.ACTIVE_STORIES,
    dynamicQueryStreamKeyCreator = StoryKeyCreator(
        targetType = targetType,
        targetId = targetId,
        sortOption = sortOption
    )
) {

    override fun provideReactorPublisher(): Flowable<AmityStory> {
        return StoryRepository().getLatestStories(
            targetType = targetType,
            targetId = targetId,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = AmityNonce.ACTIVE_STORIES
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<StoryQueryDto> {
        return getRequest()
    }

    override fun getFetchByTokenRequest(token: String): Single<StoryQueryDto> {
        return getRequest()
    }

    override fun persistResponse(dto: StoryQueryDto): Completable {
        return StoryQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: StoryQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = null,
            previous = null,
            primaryKeys = getPrimaryKeys(dto)
        )
    }

    private fun getPrimaryKeys(dto: StoryQueryDto): List<String> {
        val primaryKeys = mutableSetOf<String>()
        primaryKeys.addAll(StoryRepository().findCache(targetType, targetId).blockingGet())
        primaryKeys.addAll(
            dto.stories?.map {
                val storyId = it.storyId ?: "null"
                StoryLocalDataStore().getStoryUniqueId(storyId) ?: storyId
            } ?: emptyList()
        )
        return primaryKeys.toList()
    }

    private fun getRequest(
    ): Single<StoryQueryDto> {
        return StoryRemoteDataStore().getStories(
            targetType = targetType,
            targetId = targetId,
            orderBy = if (sortOption == AmityStorySortOption.FIRST_CREATED) "asc" else "desc",
        ).toRx2()
    }

}