package com.amity.socialcloud.sdk.infra.upload

import android.net.Uri
import co.amity.rxupload.extension.upload
import com.amity.socialcloud.sdk.core.data.file.RawFileModelMapper
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityFileAccessType
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.util.AppContext
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

class AmityFileUploadService private constructor(
    val uri: Uri,
    val uploadId: String?
) : AmityUploadService<AmityUploadResult<AmityFile>>() {

    override fun getUploadParams(): Map<String, Any> {
        return mapOf(
            AmityFileAccessType.apiKey to getAccessType().apiKey
        )
    }

    override fun getUploadHeaders(): Map<String, Any> {
        return mapOf()
    }

    override fun makeUploadServiceRequest(): Flowable<AmityUploadResult<AmityFile>> {
        return uri.upload(
            context = AppContext.get(),
            path = UPLOAD_FILE_PATH,
            params = getUploadParams(),
            headers = getUploadHeaders(),
            id = uploadId,
            accessType = getAccessType().apiKey,
        )
            .flatMap { fileProps ->
                if (fileProps.progress == 100) {
                    Single.fromCallable {
                        val fileEntity = parseEkoFileEntity(fileProps.responseBody)
                        fileEntity.filePath = fileProps.uri.path
                        val fileDao = UserDatabase.get().fileDao()
                        fileDao.insert(fileEntity)
                        RawFileModelMapper().map(fileEntity)
                    }
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .flatMapPublisher {
                            Flowable.just(
                                AmityUploadResult.COMPLETE(
                                    AmityFile(it)
                                )
                            )
                        }
                } else {
                    Flowable.just(AmityUploadResult.PROGRESS(AmityUploadInfo(fileProps)))
                }
            }
            .onErrorReturn {
                val exception = parseErrorResponse(it)
                AmityUploadResult.ERROR(exception)
            }
    }

    class Builder {

        private lateinit var uri: Uri
        private var uploadId: String? = null

        internal fun fileUri(uri: Uri): Builder {
            this.uri = uri
            return this
        }

        fun uploadId(uploadId: String): Builder {
            this.uploadId = uploadId
            return this
        }

        fun build(): AmityFileUploadService {
            return AmityFileUploadService(uri, uploadId)
        }
    }
}