package com.amity.socialcloud.sdk.core.data.reaction

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.EkoReactionDao
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Maybe
import org.amity.types.ObjectId
import org.joda.time.DateTime

internal class ReactionLocalDataStore {

    fun saveReactionsAndDeletePrevious(
        referenceType: AmityReactionReferenceType,
        ids: List<String>,
        reactions: List<EkoReactionEntity>
    ): Completable {
        return Completable.fromAction {
            val reactionDao: EkoReactionDao = UserDatabase.get().reactionDao()
            ids.forEach {
                reactionDao.deleteByReferenceIdAndUserId(
                    it,
                    referenceType.value,
                    SessionLocalDataStore().getActiveUserId()
                )
            }
            reactionDao.insert(reactions)
        }
    }

    fun saveReactions(
        reactions: List<EkoReactionEntity>
    ): Completable {
        return Completable.fromAction {
            val reactionDao: EkoReactionDao = UserDatabase.get().reactionDao()
            reactionDao.insert(reactions)
        }
    }

    fun addReaction(
        reactionId: String,
        referenceType: AmityReactionReferenceType,
        referenceId: String,
        reactionName: String,
        userId: String,
        userDisplayName: String? = null,
        createdAt: DateTime? = null,
        updateAt: DateTime? = null,
    ): Completable {
        return Completable.fromAction {
            val reactionDao = UserDatabase.get().reactionDao()
            val reactionEntity =
                EkoReactionEntity.createReaction(
                    reactionId,
                    referenceType,
                    referenceId,
                    reactionName,
                    userId,
                    createdAt,
                    updateAt
                )
            reactionDao.insert(reactionEntity)
        }
    }

    fun removeReaction(
        referenceType: AmityReactionReferenceType,
        referenceId: String,
        reactionName: String,
        userId: String,
    ): Completable {
        return Completable.fromAction {
            val reactionDao = UserDatabase.get().reactionDao()
            reactionDao.deleteByPrimaryKey(
                referenceType.value,
                referenceId,
                userId,
                reactionName
            )
        }
    }

    fun getReactionsPagingSource(
        referenceType: AmityReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): PagingSource<Int, EkoReactionEntity> {
        return UserDatabase.get().reactionPagingDao().getReactionPagingSource(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName
        )
    }

    fun getLatestReaction(
        referenceType: AmityReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): Flowable<EkoReactionEntity> {
        return UserDatabase.get().reactionDao()
            .getLatestReaction(referenceType.value, referenceId, reactionName, DateTime.now())
    }

    fun getReaction(referenceType: AmityReactionReferenceType,
                    referenceId: String,
                    reactionName: String,
                    userId: String): Maybe<EkoReactionEntity> {
        return UserDatabase.get().reactionDao()
            .getReactionByReferenceAndReactionName(referenceType.value,
                referenceId,
                reactionName,
                userId)
    }

    fun createReaction(referenceType: AmityReactionReferenceType,
                       referenceId: String,
                       reactionName: String,
                       userId: String): EkoReactionEntity {
        return EkoReactionEntity.createReaction(
            EkoReactionEntity.REACTION_PREFIX + ObjectId.get().toHexString(),
            referenceType,
            referenceId,
            reactionName,
            userId,
            null,
            null
        )
    }

    fun addReaction(reaction: EkoReactionEntity) : Completable {
        return Completable.fromAction {
            val userDatabase = UserDatabase.get()
            val reactionDao = userDatabase.reactionDao()
            userDatabase.runInTransaction {
                reactionDao.insert(reaction)
                when (AmityReactionReferenceType.enumOf(reaction.referenceType)) {
                    AmityReactionReferenceType.POST -> {
                        val postDao = userDatabase.postDao()
                        val post = postDao.getByIdNow(reaction.referenceId)
                        if (post != null) {
                            val reactionMap = post.reactions
                            post.reactionCount = post.reactionCount + 1
                            reactionMap[reaction.reactionName] = post.reactions.getCount(reaction.reactionName) + 1
                            post.reactions = reactionMap
                            postDao.update(post)
                        }
                    }
                    AmityReactionReferenceType.MESSAGE -> {
                        val messageDao = userDatabase.messageDao()
                        val message = messageDao.getByMessageIdNow(reaction.referenceId)
                        if (message != null) {
                            val reactionMap = message.reactions
                            message.reactionCount = message.reactionCount + 1
                            reactionMap[reaction.reactionName] = message.reactions.getCount(reaction.reactionName) + 1
                            message.reactions = reactionMap
                            messageDao.update(message)
                        }
                    }
                    AmityReactionReferenceType.STORY -> {
                        val storyDao = userDatabase.storyDao()
                        val story = storyDao.getByStoryIdNow(reaction.referenceId)
                        if (story != null) {
                            val reactionMap = story.reactions
                            story.reactionCount = story.reactionCount + 1
                            reactionMap[reaction.reactionName] = story.reactions.getCount(reaction.reactionName) + 1
                            story.reactions = reactionMap
                            storyDao.update(story)
                        }
                    }
                    AmityReactionReferenceType.COMMENT -> {
                        val commentDao = userDatabase.commentDao()
                        val comment = commentDao.getByIdNow(reaction.referenceId)
                        if (comment != null) {
                            val reactionMap = comment.reactions
                            comment.reactionCount = comment.reactionCount + 1
                            reactionMap[reaction.reactionName] = comment.reactions.getCount(reaction.reactionName) + 1
                            comment.reactions = reactionMap
                            commentDao.update(comment)

                            if (comment.referenceType == "post") {
                                // dummy update post
                                val postDao = userDatabase.postDao()
                                postDao.updatePost(comment.referenceId)
                            }
                        }
                    }
                }
            }
        }
    }

    fun removeReaction(reaction: EkoReactionEntity) : Completable {
        return Completable.fromAction {
            val userDatabase = UserDatabase.get()
            val reactionDao = userDatabase.reactionDao()

            userDatabase.runInTransaction {
                val reactionName = reaction.reactionName
                reactionDao.delete(reaction)
                when (AmityReactionReferenceType.enumOf(reaction.referenceType)) {
                    AmityReactionReferenceType.POST -> {
                        val postDao = userDatabase.postDao()
                        val post = postDao.getByIdNow(reaction.referenceId)
                        if (post != null) {
                            val reactionMap = post.reactions
                            val totalReactionCount = Math.max(0, post.reactionCount - 1)
                            val reactionCount = Math.max(0, post.reactions.getCount(reactionName) - 1)
                            post.reactionCount = totalReactionCount
                            reactionMap[reactionName] = reactionCount
                            post.reactions = reactionMap
                            postDao.update(post)
                        }
                    }
                    AmityReactionReferenceType.MESSAGE -> {
                        val messageDao = userDatabase.messageDao()
                        val message = messageDao.getByMessageIdNow(reaction.referenceId)
                        if (message != null) {
                            val reactionMap = message.reactions
                            val totalReactionCount = Math.max(0, message.reactionCount - 1)
                            val reactionCount = Math.max(0, message.reactions.getCount(reactionName) - 1)
                            message.reactionCount = totalReactionCount
                            reactionMap[reactionName] = reactionCount
                            message.reactions = reactionMap
                            messageDao.update(message)
                        }
                    }
                    AmityReactionReferenceType.STORY -> {
                        val storyDao = userDatabase.storyDao()
                        val story = storyDao.getByStoryIdNow(reaction.referenceId)
                        if (story != null) {
                            val reactionMap = story.reactions
                            val totalReactionCount = Math.max(0, story.reactionCount - 1)
                            val reactionCount = Math.max(0, story.reactions.getCount(reactionName) - 1)
                            story.reactionCount = totalReactionCount
                            reactionMap[reactionName] = reactionCount
                            story.reactions = reactionMap
                            storyDao.update(story)
                        }
                    }
                    AmityReactionReferenceType.COMMENT -> {
                        val commentDao = userDatabase.commentDao()
                        val comment = commentDao.getByIdNow(reaction.referenceId)
                        if (comment != null) {
                            val reactionMap = comment.reactions
                            val totalReactionCount = Math.max(0, comment.reactionCount - 1)
                            val reactionCount = Math.max(0, comment.reactions.getCount(reactionName) - 1)
                            comment.reactionCount = totalReactionCount
                            reactionMap[reactionName] = reactionCount
                            comment.reactions = reactionMap
                            commentDao.update(comment)

                            if (comment.referenceType == "post") {
                                // dummy update post
                                val postDao = userDatabase.postDao()
                                postDao.updatePost(comment.referenceId)
                            }
                        }
                    }
                }
            }
        }
    }

}