package com.amity.socialcloud.sdk.social.data.storytarget.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.social.storytarget.AmityGlobalStoryTargetsQueryOption
import com.amity.socialcloud.sdk.dto.social.story.StoryQueryDto
import com.amity.socialcloud.sdk.dto.social.story.StoryTargetDto
import com.amity.socialcloud.sdk.entity.social.story.StoryTargetEntity
import com.amity.socialcloud.sdk.model.social.story.AmityStoryTarget
import com.amity.socialcloud.sdk.social.data.story.StoryQueryPersister
import com.amity.socialcloud.sdk.social.data.storytarget.StoryTargetLocalDataStore
import com.amity.socialcloud.sdk.social.data.storytarget.StoryTargetRemoteDataStore
import com.amity.socialcloud.sdk.social.data.storytarget.StoryTargetRepository
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single
import org.amity.types.ObjectId
import org.joda.time.DateTime


internal class StoryTargetMediator(
    private val option: AmityGlobalStoryTargetsQueryOption
) : DynamicQueryStreamMediator<StoryTargetEntity, StoryQueryDto, AmityStoryTarget>(
    nonce = AmityNonce.STORY_TARGET_PAGE,
    dynamicQueryStreamKeyCreator = StoryTargetKeyCreator(
        option = option
    )
) {

    private var hasLoadedUnSeen = false

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityStoryTarget> {
        return StoryTargetRepository().getLatestStoryTargets(
            option,
            dynamicQueryStreamKeyCreator,
            nonce
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<StoryQueryDto> {
        return getRequest(limit = pageSize)

    }

    override fun getFetchByTokenRequest(token: String): Single<StoryQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: StoryQueryDto): Completable {
        return StoryQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: StoryQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = if(option == AmityGlobalStoryTargetsQueryOption.SMART && dto.paging?.next == null && !hasLoadedUnSeen) AmityGlobalStoryTargetsQueryOption.SMART.apiKey else dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = getPrimaryKeys(dto.paging?.previous, dto.storyTargets)
        )
    }

    private fun getPrimaryKeys(previousToken: String?, storyTargets: List<StoryTargetDto>?) : List<String> {
        val primaryKeys = mutableSetOf<String>()
        val shouldAddCacheKeys = previousToken == null && !hasLoadedUnSeen && option != AmityGlobalStoryTargetsQueryOption.SEEN
        if(shouldAddCacheKeys) {
            val cacheKeys = StoryTargetLocalDataStore().findCacheKeys(
                option = option,
                expiresAfter = storyTargets?.firstOrNull()?.lastStoryExpiresAt ?: DateTime.now()
            )
            primaryKeys.addAll(cacheKeys)
        }
        primaryKeys.addAll(storyTargets?.map {
            AmityStoryTarget.generateUniqueId(
                it.targetType ?: "unknown", it.targetPublicId ?: ObjectId.get().toHexString()
            )
        } ?: emptyList())

        return primaryKeys.toList()
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<StoryQueryDto> {
        if(token == AmityGlobalStoryTargetsQueryOption.SMART.apiKey) {
            hasLoadedUnSeen = true
            return getFirstPageRequest(StoryTargetRepository().getDefaultPageSize())
        }

        val queryOption = when (option) {
            AmityGlobalStoryTargetsQueryOption.SEEN -> AmityGlobalStoryTargetsQueryOption.SEEN.apiKey
            AmityGlobalStoryTargetsQueryOption.UNSEEN -> AmityGlobalStoryTargetsQueryOption.UNSEEN.apiKey
            AmityGlobalStoryTargetsQueryOption.ALL -> AmityGlobalStoryTargetsQueryOption.ALL.apiKey
            else -> if (hasLoadedUnSeen) AmityGlobalStoryTargetsQueryOption.SEEN.apiKey else AmityGlobalStoryTargetsQueryOption.UNSEEN.apiKey
        }

        return StoryTargetRemoteDataStore().getGlobalFeed(
            seenState = queryOption,
            limit = limit,
            token = token
        ).toRx2()
    }

}