package com.amity.socialcloud.sdk.core.data.reaction

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.core.data.reaction.paging.ReactionMediator
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.core.session.eventbus.LiveReactionEventBus
import com.amity.socialcloud.sdk.model.core.reaction.AmityLiveReactionReferenceType
import com.amity.socialcloud.sdk.model.core.reaction.AmityReaction
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.amity.socialcloud.sdk.model.core.reaction.live.AmityLiveReaction
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Maybe
import io.reactivex.rxjava3.schedulers.Schedulers

internal class ReactionRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    @OptIn(ExperimentalPagingApi::class)
    fun getReactionPagingData(
        referenceType: AmityReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): Flowable<PagingData<AmityReaction>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = ReactionMediator(
                referenceType = referenceType,
                referenceId = referenceId,
                reactionName = reactionName
            ),
            pagingSourceFactory = {
                ReactionLocalDataStore().getReactionsPagingSource(
                    referenceType = referenceType,
                    referenceId = referenceId,
                    reactionName = reactionName
                )
            },
            modelMapper = ReactionModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getMyReactions(referenceType: AmityReactionReferenceType, referenceId: String): List<String> {
        var list = emptyList<String>()
        val reactionDao = UserDatabase.get().reactionDao()
        val entities = reactionDao.getAllMyReactionsByReferenceTypeAndReferenceIdNow(
            referenceType.value,
            referenceId
        )
        if (entities.isNotEmpty()) {
            list = entities.map {
                it.reactionName
            }
        }
        return list
    }

    fun getLatestReaction(
        referenceType: AmityReactionReferenceType,
        referenceId: String,
        reactionName: String?
    ): Flowable<AmityReaction> {
        return ReactionLocalDataStore().getLatestReaction(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName
        ).map {
            ReactionModelMapper().map(it)
        }
    }

    fun addReaction(referenceType: AmityReactionReferenceType, referenceId: String, reactionName: String) : Completable {
        return getMyReaction(referenceType, referenceId, reactionName)
            .subscribeOn(Schedulers.io())
            .isEmpty
            .flatMapCompletable { isEmpty: Boolean ->
                if (isEmpty) {
                    val reaction = createMyReaction(referenceType, referenceId, reactionName)
                    ReactionLocalDataStore().addReaction(reaction)
                        .concatWith(
                            ReactionRemoteDataStore().addReaction(referenceType, referenceId, reactionName)
                                .ignoreElement()
                                .onErrorResumeNext {
                                    // revert local value
                                    ReactionLocalDataStore().removeReaction(reaction)
                                }
                        )
                } else {
                    Completable.complete()
                }
            }
    }

    fun removeReaction(referenceType: AmityReactionReferenceType, referenceId: String, reactionName: String) : Completable {
        return getMyReaction(referenceType, referenceId, reactionName)
            .subscribeOn(Schedulers.io())
            .flatMapCompletable { removingReaction: EkoReactionEntity ->
                ReactionLocalDataStore().removeReaction(removingReaction)
                    .concatWith(
                        ReactionRemoteDataStore().removeReaction(referenceType, referenceId, reactionName)
                            .ignoreElement()
                            .onErrorResumeNext {
                                // revert local value
                                ReactionLocalDataStore().addReaction(removingReaction)
                            }
                    )
            }
    }

    private fun getMyReaction(referenceType: AmityReactionReferenceType, referenceId: String, reactionName: String) : Maybe<EkoReactionEntity> {
        return ReactionLocalDataStore().getReaction(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName,
            userId = SessionLocalDataStore().getActiveUserId()
        )
    }

    fun createMyReaction(referenceType: AmityReactionReferenceType, referenceId: String, reactionName: String) : EkoReactionEntity {
        return ReactionLocalDataStore().createReaction(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName,
            userId =  SessionLocalDataStore().getActiveUserId()
        )
    }

    fun observeLiveReactions(
        referenceId: String,
        referenceType: AmityLiveReactionReferenceType,
    ): Flowable<List<AmityLiveReaction>> {
        return LiveReactionEventBus
            .observe()
            .map { reactions ->
                reactions.filter {
                    it.getReferenceType().let(AmityLiveReactionReferenceType::enumOf) == referenceType && it.getReferenceId() == referenceId
                }
            }
    }

    fun addLiveReactions(
        liveStreamId: String,
        reactions: List<AmityLiveReaction>
    ): Completable {
        return ReactionRemoteDataStore().addLiveReactions(liveStreamId, reactions)
            .ignoreElement()
    }

}