package com.ekoapp.ekosdk.internal.data.dao;

import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.api.core.AmityCoreClient;
import com.amity.socialcloud.sdk.api.social.member.query.AmityCommunityMembershipSortOption;
import com.amity.socialcloud.sdk.model.core.role.AmityRoles;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.CommunityMembershipPermissionEntity;
import com.ekoapp.ekosdk.internal.data.model.CommunityMembershipRoleEntity;
import com.ekoapp.ekosdk.internal.entity.ChannelMembershipEntity;
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity;
import com.google.common.base.Objects;

import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;

@Dao
public abstract class EkoCommunityMembershipDao extends EkoObjectDao<CommunityMembershipEntity> {

    private final EkoCommunityDao communityDao;
    private final EkoCommunityRoleDao communityRoleDao;
    private final EkoCommunityPermissionDao communityPermissionDao;

    public EkoCommunityMembershipDao() {
        UserDatabase db = UserDatabase.get();
        communityRoleDao = db.communityRoleDao();
        communityPermissionDao = db.communityPermissionDao();
        communityDao = db.communityDao();
    }

    @Delete
    public abstract void delete(CommunityMembershipEntity communityMembership);

    @Query("DELETE from community_membership")
    public abstract void deleteAll();

    @Query("DELETE from community_membership where community_membership.communityId = :communityId")
    abstract void deleteByCommunityIdImpl(String communityId);

    @Query("SELECT * from community_membership where communityId = :communityId and userId = :userId LIMIT 1")
    abstract CommunityMembershipEntity getByCommunityIdAndUserIdNowImpl(String communityId, String userId);

    public CommunityMembershipEntity getByCommunityIdAndUserIdNow(String communityId, String userId) {
        return getByCommunityIdAndUserIdNowImpl(communityId, userId);
    }

    @Query("SELECT * from community_membership where id = :id LIMIT 1")
    abstract CommunityMembershipEntity getByIdNowImpl(String id);

    @Override
    public CommunityMembershipEntity getByIdNow(String id) {
        return getByIdNowImpl(id);
    }

    @Query("SELECT *" +
            " from community_membership" +
            " where communityId = :communityId" +
            " and userId = :userId" +
            " LIMIT 1")
    abstract Flowable<CommunityMembershipEntity> getByIdImpl(String communityId, String userId);

    public Flowable<CommunityMembershipEntity> getById(String communityId, String userId) {
        return getByIdImpl(communityId, userId);
    }

    @Query("SELECT * from community_membership where userId in (:ids)")
    abstract List<CommunityMembershipEntity> getByIdsNowImpl(@NotNull List<String> ids);

    @Override
    public List<CommunityMembershipEntity> getByIdsNow(@NotNull List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Transaction
    public void deleteByCommunityId(String communityId) {
        deleteByCommunityIdImpl(communityId);
    }


    @Transaction //dummy update, for triggering user update tied to community member.
    public void updateUser(String userId) {
        updateUserImpl(userId);
    }

    @Query("UPDATE community_membership set userId = :userId where userId = :userId")
    abstract void updateUserImpl(String userId);


    @Transaction
    @Override
    public void insert(CommunityMembershipEntity membership) {
        super.insert(membership);
        EkoRoleDao.update(membership, communityRoleDao, CommunityMembershipRoleEntity::create);
        EkoPermissionDao.update(membership, communityPermissionDao, CommunityMembershipPermissionEntity::create);
        updateIsJoined(membership);
    }

    @Transaction
    @Override
    public void insert(List<? extends CommunityMembershipEntity> memberships) {
        super.insert(memberships);
        EkoRoleDao.update(memberships, communityRoleDao, CommunityMembershipRoleEntity::create);
        EkoPermissionDao.update(memberships, communityPermissionDao, CommunityMembershipPermissionEntity::create);
        for (CommunityMembershipEntity membership : memberships) {
            updateIsJoined(membership);
        }
    }

    @Override
    public void update(CommunityMembershipEntity membership) {
        super.update(membership);
        EkoRoleDao.update(membership, communityRoleDao, CommunityMembershipRoleEntity::create);
        EkoPermissionDao.update(membership, communityPermissionDao, CommunityMembershipPermissionEntity::create);
        updateIsJoined(membership);
    }

    private void updateIsJoined(CommunityMembershipEntity membership) {
        if (Objects.equal(membership.getUserId(), AmityCoreClient.INSTANCE.getUserId())) {
            communityDao.updateIsJoined(membership.getCommunityId());
        }
    }

    @Query("SELECT *" +
            " from community_membership" +
            " where community_membership.communityId = :communityId" +
            " and community_membership.communityMembership in (:memberships)" +
            " and case when :isFilterByRoles then community_membership.userId in (SELECT userId from community_role where communityId = :communityId and roleName in (:roles))" +
            " else userId is not null end" + // always true
            " and community_membership.updatedAt > :now" +
            " and community_membership.userId not in " +
            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +
            " order by community_membership.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<CommunityMembershipEntity> getLatestCommunityMembershipImpl(String communityId,
                                                                                  String[] memberships,
                                                                                  Boolean isFilterByRoles,
                                                                                  String[] roles,
                                                                                  int hash,
                                                                                  int nonce,
                                                                                  DateTime now);

    public Flowable<CommunityMembershipEntity> getLatestCommunityMembership(
            String communityId,
            List<String> memberships,
            AmityRoles roles,
            int hash,
            int nonce,
            DateTime now
    ) {
        return getLatestCommunityMembershipImpl(
                communityId,
                memberships.toArray(new String[0]),
                !roles.isEmpty(),
                roles.get().toArray(new String[0]),
                hash,
                nonce,
                now
        );
    }

}
