package com.amity.socialcloud.sdk.chat.data.subchannel

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.chat.data.marker.subchannel.SubChannelMarkerRepository
import com.amity.socialcloud.sdk.chat.data.subchannel.paging.SubChannelMediator
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.MarkerSyncEngine
import com.amity.socialcloud.sdk.core.data.tombstone.TombstoneRepository
import com.amity.socialcloud.sdk.model.chat.subchannel.AmitySubChannel
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.core.CoreClient
import com.ekoapp.ekosdk.internal.SubChannelEntity
import com.ekoapp.ekosdk.internal.TombstoneModelType
import com.ekoapp.ekosdk.internal.api.dto.SubChannelListDto
import com.ekoapp.ekosdk.internal.data.model.MessagePreviewEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class SubChannelRepository : AmityObjectRepository<SubChannelEntity, AmitySubChannel>() {

    override fun fetchAndSave(objectId: String): Completable {
        return SubChannelRemoteDataStore().fetchSubChannel(objectId)
            .flatMapCompletable { dto ->
                fetchSubChannelMarker(dto)
                    .andThen(persistSubChannels(dto))
            }
            .onErrorResumeNext {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    SubChannelLocalDataStore().hardDelete(objectId).andThen(Completable.error(it))
                } else {
                    Completable.error(it)
                }
            }
    }

    override fun queryFromCache(objectId: String): SubChannelEntity? {
        return SubChannelLocalDataStore().getSubChannel(objectId)
    }

    override fun mapper(): ModelMapper<SubChannelEntity, AmitySubChannel> {
        return SubChannelModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<SubChannelEntity> {
        return SubChannelLocalDataStore().observe(objectId)
    }

    fun createSubChannel(channelId: String, name: String): Single<AmitySubChannel> {
        return SubChannelRemoteDataStore().createSubChannel(
            channelId = channelId,
            name = name
        ).flatMap { dto ->
            fetchSubChannelMarker(dto)
                .andThen(persistAndReturnSubChannels(dto))
        }
    }

    fun updateSubChannel(subChannelId: String, name: String): Completable {
        return SubChannelRemoteDataStore().updateSubChannel(subChannelId, name)
            .flatMapCompletable {
                persistSubChannels(it)
            }
    }

    fun deleteSubChannel(subChannelId: String, hardDelete: Boolean): Completable {
        return SubChannelRemoteDataStore().deleteSubChannel(subChannelId, hardDelete)
            .flatMap {
                persistSubChannels(it).andThen(Single.just(it.subChannels.first()))
            }.flatMapCompletable {
                if (it.isDeleted == false) {
                    Completable.complete()
                } else if (hardDelete) {
                    SubChannelLocalDataStore().hardDelete(subChannelId).andThen(Completable.defer {
                        TombstoneRepository().saveTombstone(
                            objectId = subChannelId,
                            tombstoneModelType = TombstoneModelType.SUB_CHANNEL,
                            amityError = AmityError.ITEM_NOT_FOUND
                        )
                    })
                } else {
                    SubChannelLocalDataStore().softDelete(subChannelId)
                }
            }
    }

    fun getSubChannelPagingData(
        channelId: String,
        excludeMainSubChannel: Boolean,
        isDeleted: Boolean
    ): Flowable<PagingData<AmitySubChannel>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = SubChannelMediator(
                channelId = channelId,
                excludeMainSubChannel = excludeMainSubChannel,
                isDeleted = isDeleted
            ),
            pagingSourceFactory = {
                SubChannelLocalDataStore().getSubChannelPagingSource(
                    channelId = channelId,
                    isDeleted = isDeleted
                )
            },
            modelMapper = SubChannelModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getLatestSubChannel(
        isDeleted: Boolean,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmitySubChannel> {
        return SubChannelLocalDataStore().getLatestSubChannel(
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).map {
            SubChannelModelMapper().map(it)
        }
    }

    internal fun getSubChannel(subChannelId: String): Single<AmitySubChannel> {
        return SubChannelLocalDataStore().observe(subChannelId)
            .firstOrError()
            .map {
                SubChannelModelMapper().map(it)
            }
    }

    internal fun persistSubChannels(dto: SubChannelListDto): Completable {
        return SubChannelQueryPersister().persist(dto)
    }

    internal fun persistAndReturnSubChannels(dto: SubChannelListDto): Single<AmitySubChannel> {
        return persistSubChannels(dto).andThen(getSubChannel(dto.subChannels.first().subChannelId!!))
    }
    
    internal fun updateMarkerHash(subChannelId: String, hash: Int) {
        SubChannelLocalDataStore().updateMarkerHash(subChannelId, hash)
    }
    
    internal fun updateUserMarkerHash(subChannelId: String, hash: Int) {
        SubChannelLocalDataStore().updateUserMarkerHash(subChannelId, hash)
    }
    
    internal fun notifySubChannelChanges(subChannelIds: List<String>) {
        SubChannelLocalDataStore().notifySubChannelsChanges(subChannelIds)
    }
    
    private fun fetchSubChannelMarker(dto: SubChannelListDto): Completable {
        return if (CoreClient.isUnreadCountEnable()) {
            dto.subChannels
                .filter {
                    MarkerSyncEngine.isMarkerSyncSupport(it.channelType)
                }
                .mapNotNull { it.subChannelId }
                .let { subChannelIds ->
                    when {
                        subChannelIds.isEmpty() -> Completable.complete()
                        else -> SubChannelMarkerRepository().fetchSubChannelMarker(subChannelIds)
                    }
                }
                .onErrorComplete()
            } else {
            Completable.complete()
        }
    }
    
    internal fun updateMessagePreview(subChannelId: String, messagePreviewId: String): Completable {
        return SubChannelLocalDataStore().updateMessagePreview(subChannelId, messagePreviewId)
    }
    
    internal fun notifyChanges(subChannelId: String) {
        SubChannelLocalDataStore().notifyChanges(subChannelId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }
}