package com.amity.socialcloud.sdk.chat.data.message

import android.net.Uri
import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.api.chat.message.query.AmityMessageQuerySortOption
import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class MessageLocalDataStore {

    fun saveMessages(
        messages: List<EkoMessageEntity>,
    ): Completable {
        return Completable.fromAction {
            val refMessages = messages.filter { it.messageId != it.uniqueId  }
            val noRefMessages = messages.filter { it.messageId == it.uniqueId }
            val savingMessages = mutableListOf<EkoMessageEntity>()
            for(message in noRefMessages) {
                val cachedObject = UserDatabase.get().messageDao().getByMessageIdNow(message.messageId)
                if(cachedObject != null) {
                    message.uniqueId = cachedObject.uniqueId
                }
                savingMessages.add(message)
            }
            for (message in refMessages) {
                val cachedObject = UserDatabase.get().messageDao().getByIdNow(message.uniqueId)
                if(cachedObject != null) {
                    // Device without sim card could have unreliable device DateTime
                    // TODO: Find a better way to resolve this
                    if(cachedObject.uniqueId == cachedObject.messageId
                        && message.updatedAt?.isBefore(cachedObject.updatedAt) ?: true) {
                        message.createdAt = cachedObject.createdAt
                        message.updatedAt = cachedObject.updatedAt
                    }
                } else {
                    message.uniqueId = message.messageId
                }
                savingMessages.add(message)
            }
            UserDatabase.get().messageDao().save(savingMessages)
        }.subscribeOn(Schedulers.io())
    }

    fun createMessage(
        subChannelId: String,
        parentId: String?,
        type: String,
        data: JsonObject?,
        metadata: JsonObject?,
        tags: AmityTags,
        fileUri: Uri?,
        mentionees: List<AmityMentioneeTarget>
    ): Single<EkoMessageEntity> {

        return Single.fromCallable {
            val message = EkoMessageEntity()
            message.subChannelId = subChannelId
            message.messageId = message.uniqueId
            message.userId = AmityCoreClient.getUserId()
            message.parentId = parentId
            message.type = type
            message.data = data?.deepCopy()
            message.metadata = metadata
            message.setTags(tags)
            message.mentionees = MessageMentionMapper().map(mentionees)
            message.setState(AmityMessage.State.CREATED)
            val now = DateTime.now()
            message.createdAt = now
            message.updatedAt = now
            message.editedAt = now

            fileUri?.let {
                message.data?.addProperty("fileId", message.messageId)
            }

            val messageDao = UserDatabase.get().messageDao()
            val highestChannelSegment = messageDao.getHighestSegment(message.subChannelId) + 1
            message.setChannelSegment(highestChannelSegment + 1)
            messageDao.insert(message)
            message
        }
            .subscribeOn(Schedulers.io())

    }

    fun getMessage(messageId: String): EkoMessageEntity? {
        return UserDatabase.get().messageDao().getByMessageIdNow(messageId)
    }
    
    fun getMessageUniqueId(messageId: String): String? {
        return UserDatabase.get().messageDao().getUniqueIdByMessageId(messageId)
    }

    fun updateMessageState(messageId: String, state: AmityMessage.State) : Completable {
        return UserDatabase.get().messageDao().updateSyncState(messageId, state.stateName)
            .subscribeOn(Schedulers.io())
    }

    fun observeLatestMessage(subChannelId: String, isDeleted: Boolean?): Flowable<EkoMessageEntity> {
        return  UserDatabase.get().messageDao().getLatestMessage(subChannelId, isDeleted)
    }

    fun observeMessage(messageId: String) : Flowable<EkoMessageEntity> {
        return UserDatabase.get().messageDao().getById(messageId)
    }

    fun hardDeleteAllFromSubChannel(subChannelId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().messageDao().hardDeleteAllFromChannel(subChannelId)
        }
    }

    fun hardDeleteMessage(messageId: String): Completable {
        return UserDatabase.get().messageDao().hardDeleteMessage(messageId)
    }

    fun observeMessages(
        channelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?,
        sortOption: AmityMessageQuerySortOption
    ): Flowable<List<EkoMessageEntity>> {
        return UserDatabase.get().messageDao().observeMessages(
            channelId,
            isFilterByParentId,
            parentId,
            includingTags,
            excludingTags,
            isDeleted,
            type?.let(::listOf) ?: listOf(),
            sortOption
        )
    }

    fun getMessagePagingSource(
        subChannelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        sortOption: AmityMessageQuerySortOption,
        dataType: String?,
        aroundMessageId: String?,
        uniqueId: String? = null,
    ): PagingSource<Int, EkoMessageEntity> {
        return UserDatabase.get().messagePagingDao().getMessagePagingSource(
            subChannelId = subChannelId,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            sortOption = sortOption,
            dataType = dataType,
            aroundMessageId = aroundMessageId,
            uniqueId = uniqueId,
        )
    }

    fun getLatestMessage(
        subChannelId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        type: String?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        isUnsyncedOnly: Boolean = false,
    ) : Flowable<EkoMessageEntity> {
        return if (isUnsyncedOnly) {
            UserDatabase.get().messageDao().getLatestUnsyncMessage(
                    subChannelId,
                    isFilterByParentId,
                    parentId,
                    includingTags.toTypedArray(),
                    excludingTags.toTypedArray(),
                    isDeleted,
                    type?.let(::listOf) ?: listOf(),
                    dynamicQueryStreamKeyCreator.toMap().hashCode(),
                    nonce
            )
        } else {
            UserDatabase.get().messageDao().getLatestMessage(
                    subChannelId,
                    isFilterByParentId,
                    parentId,
                    includingTags.toTypedArray(),
                    excludingTags.toTypedArray(),
                    isDeleted,
                    type?.let(::listOf) ?: listOf(),
                    dynamicQueryStreamKeyCreator.toMap().hashCode(),
                    nonce,
                    DateTime.now()
            )
        }
        
    }

    fun notifyUserUpdate(userId: String) {
        UserDatabase.get().messageDao().updateUser(userId)
    }

    fun hasInLocal(messageId: String): Boolean {
        var hasInLocal = false
        Completable.fromCallable {
            val messageDao = UserDatabase.get().messageDao()
            val entity = messageDao.getByMessageIdNow(messageId)
            if (entity != null) {
                hasInLocal = true
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return hasInLocal
    }
    
    fun updateMarkerHash(messageId: String, hash: Int) {
        UserDatabase.get().messageDao().updateMarkerHash(messageId, hash)
    }
    
    fun cleanUpFailedMessages() {
        UserDatabase.get().messageDao().cleanUpFailedMessages()
    }

    fun findCacheAroundMessage(messageId: String): Single<List<String>> {
        return Single.fromCallable {
            val message = UserDatabase.get().messageDao().getByMessageIdNow(messageId)
            val messageIds = mutableListOf<String>()
            if(message != null) {
                messageIds.addAll(UserDatabase.get().messageDao().getCacheBeforeMessageId(messageId).map { it.messageId })
                messageIds.add(message.messageId)
                messageIds.addAll(UserDatabase.get().messageDao().getCacheAfterMessageId(messageId).map { it.messageId })
            }
            messageIds
        }
    }

}