package com.amity.socialcloud.sdk.social.data.post.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostQueryPersister
import com.amity.socialcloud.sdk.social.data.post.PostRemoteDataStore
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.QueryStreamMediator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Single

@ExperimentalPagingApi
class GlobalFeedMediator(
    private val dataTypes: List<AmityPost.DataType>
) : QueryStreamMediator<PostEntity, EkoPostQueryDto>(
    nonce = AmityNonce.GLOBAL_FEED,
    queryStreamKeyCreator = GlobalFeedKeyCreator(dataTypes),
) {

    override fun forceRefresh(): Boolean = true

    override fun getFirstPageRequest(pageSize: Int): Single<EkoPostQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoPostQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoPostQueryDto): Completable {
        return PostQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoPostQueryDto): QueryStreamQueryToken {
        return QueryStreamQueryToken(
            queryStreamKeyCreator = GlobalFeedKeyCreator(dataTypes),
            next = dto.paging.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.posts?.map { post -> post.postId } ?: listOf()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        return PostRemoteDataStore().getGlobalFeedPosts(
            dataTypes = getDataTypes(),
            limit = limit,
            token = token
        ).toRx2()
    }

    private fun getDataTypes(): List<String>? {
        if (dataTypes.isNotEmpty()) {
            val typeArrayList = arrayListOf<String>()
            dataTypes.forEach { typeArrayList.add(it.getApiKey()) }
            return typeArrayList
        }
        return null
    }
}