package com.amity.socialcloud.sdk.chat.data.channel

import com.amity.socialcloud.sdk.api.chat.channel.create.AmityChannelCreator
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.ChannelQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelRead
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelsReadRequest
import com.ekoapp.ekosdk.internal.api.socket.request.ChannelUpdateRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CreateChannelRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CreateConversationWithUserIdsRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

internal class ChannelRemoteDataStore {

    fun createChannel(
            creationType: AmityChannelCreator.CreationType,
            displayName: String? = null,
            avatarFileId: String? = null,
            metadata: JsonObject? = null,
            tags: AmityTags? = null,
            userIds: List<String>? = null,
            isPublic: Boolean? = null,
    ): Single<ChannelQueryDto> {

        val request = CreateChannelRequest(
                type = creationType.apiKey,
                displayName = displayName,
                tags = tags?.takeIf { it.isNotEmpty() },
                metadata = metadata,
                avatarFileId = avatarFileId,
                isPublic = isPublic,
                userIds = userIds?.takeIf { it.isNotEmpty() }
        )
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.createChannel(request)
        }
    }


    fun createConversationChannel(
            displayName: String? = null,
            avatarFileId: String? = null,
            metadata: JsonObject? = null,
            tags: AmityTags? = null,
            userIds: List<String>? = null
    ): Single<ChannelQueryDto> {

        val request = CreateConversationWithUserIdsRequest(
                displayName = displayName,
                avatarFileId = avatarFileId,
                userIds = userIds?.takeIf { it.isNotEmpty() },
                tags = tags?.takeIf { it.isNotEmpty() },
                metadata = metadata,
                isDistinct = true
        )
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.createConversationChannel(request)
        }
    }

    fun updateChannel(
            channelId: String,
            displayName: String? = null,
            avatarFileId: String? = null,
            metadata: JsonObject? = null,
            tags: AmityTags? = null,
    ): Single<ChannelQueryDto> {

        val request = ChannelUpdateRequest(
                channelId = channelId,
                displayName = displayName,
                avatarFileId = avatarFileId,
                metadata = metadata,
                tags = tags
        )
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.updateChannel(channelId, request)
        }
    }

    fun getChannel(channelId: String): Single<ChannelQueryDto> {
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.getChannel(channelId)
        }
    }

    fun joinChannel(channelId: String): Single<ChannelQueryDto> {
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.joinChannel(channelId)
        }
    }

    fun leaveChannel(channelId: String): Single<ChannelQueryDto> {
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.leaveChannel(channelId)
        }
    }

    fun queryChannels(
            types: List<String>? = null,
            filter: String? = null,
            includingTags: List<String>? = null,
            excludingTags: List<String>? = null,
            isDeleted: Boolean? = null,
            sortBy: String? = null,
            options: QueryOptionsRequestParams
    ): Single<ChannelQueryDto> {
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.queryChannels(
                    types = types?.takeIf { it.isNotEmpty() },
                    filter = filter,
                    tags = includingTags?.takeIf { it.isNotEmpty() },
                    excludeTags = excludingTags?.takeIf { it.isNotEmpty() },
                    isDeleted = isDeleted,
                    sortBy = sortBy,
                    limit = options.limit,
                    token = options.token
            )
        }
    }

    fun getChannels(
            channelIds: List<String> = emptyList()
    ): Single<ChannelQueryDto> {
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.getChannels(channelIds)
        }
    }

    fun bulkMarkRead(
            channels: List<Pair<String, Int>>
    ): Single<JsonObject> {
        return AmityHttpClient.get(ChannelApi::class).flatMap {
            it.bulkMarkSeen(
                    ChannelsReadRequest(
                            channels = channels.map { channel ->
                                ChannelRead(channelId = channel.first, readToSegment = channel.second)
                            }
                    )
            )
        }
    }
}