package com.amity.socialcloud.sdk.social.data.post

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.post.query.AmityCommunityFeedSortOption
import com.amity.socialcloud.sdk.api.social.post.query.AmityUserFeedSortOption
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.tombstone.TombstoneRepository
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.model.social.post.AmityPostAttachment
import com.amity.socialcloud.sdk.social.data.post.paging.CommunityPostMediator
import com.amity.socialcloud.sdk.social.data.post.paging.CustomPostRankingMediator
import com.amity.socialcloud.sdk.social.data.post.paging.GlobalFeedMediator
import com.amity.socialcloud.sdk.social.data.post.paging.SearchPostMediator
import com.amity.socialcloud.sdk.social.data.post.paging.UserPostMediator
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.TombstoneModelType
import com.ekoapp.ekosdk.internal.api.socket.request.FlagContentRequest
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class PostRepository : AmityObjectRepository<PostEntity, AmityPost>() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    override fun fetchAndSave(objectId: String): Completable {
        return PostRemoteDataStore().fetchPost(objectId)
            .flatMapCompletable { PostQueryPersister().persist(it) }
            .onErrorResumeNext {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    PostLocalDataStore().hardDelete(objectId).andThen(Completable.error(it))
                } else {
                    Completable.error(it)
                }
            }
    }

    override fun queryFromCache(objectId: String): PostEntity? {
        return PostLocalDataStore().getPost(objectId)
    }

    override fun mapper(): ModelMapper<PostEntity, AmityPost> {
        return PostModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<PostEntity> {
        return PostLocalDataStore().observe(objectId)
    }

    fun getPostByIds(postIds: List<String>): Flowable<List<AmityPost>> {
        return PostRemoteDataStore().getPostByIds(postIds)
            .flatMap {
                PostQueryPersister().persist(it)
                    .andThen(Single.just((it.posts.map { it.postId })))
            }
            .flatMapPublisher { ids ->
                PostLocalDataStore().getPostByIds(ids)
                    .map { posts ->
                        posts.map {
                            PostModelMapper().map(it)
                        }
                    }
            }
    }

    fun searchPosts(
        query: String,
        targetType: String?,
        targetId: String?,
        postTypes: List<AmityPost.DataType>,
        matchingOnlyParentPosts: Boolean,
    ): Flowable<PagingData<AmityPost>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                initialLoadSize = (getDefaultPageSize() / 2),
                prefetchDistance = 0
            ),
            dynamicQueryStreamMediator = SearchPostMediator(
                query = query,
                targetType = targetType,
                targetId = targetId,
                matchingOnlyParentPosts = matchingOnlyParentPosts,
                postTypes = postTypes
            ),
            pagingSourceFactory = {
                PostLocalDataStore().getSearchPostsPagingSource(
                    query = query,
                    targetId = targetId,
                    targetType = targetType,
                    matchingOnlyParentPosts = matchingOnlyParentPosts,
                    postTypes = postTypes
                )
            },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getUserPostPagingData(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf(),
    ): Flowable<PagingData<AmityPost>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                initialLoadSize = (getDefaultPageSize() / 2),
                prefetchDistance = 0
            ),
            dynamicQueryStreamMediator = UserPostMediator(
                userId = userId,
                sortBy = sortOption.apiKey,
                isDeleted = isDeleted,
                postTypes = postTypes
            ),
            pagingSourceFactory = {
                PostLocalDataStore()
                    .getUserPostsPagingSource(userId, sortOption, isDeleted, postTypes)
            },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getCommunityPostPagingData(
        communityId: String,
        sortOption: AmityCommunityFeedSortOption,
        feedType: AmityFeedType,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf(),
    ): Flowable<PagingData<AmityPost>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                initialLoadSize = (getDefaultPageSize() / 2),
                prefetchDistance = 0
            ),
            dynamicQueryStreamMediator = CommunityPostMediator(
                communityId = communityId,
                sortBy = sortOption.apiKey,
                isDeleted = isDeleted,
                feedType = feedType,
                postTypes = postTypes
            ),
            pagingSourceFactory = {
                PostLocalDataStore().getCommunityPostsPagingSource(
                    communityId,
                    sortOption,
                    feedType,
                    isDeleted,
                    postTypes
                )
            },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getGlobalFeedPagingData(): Flowable<PagingData<AmityPost>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                initialLoadSize = (getDefaultPageSize() / 2),
                prefetchDistance = 0
            ),
            queryStreamMediator = GlobalFeedMediator(),
            pagingSourceFactory = { PostLocalDataStore().getGlobalPostsPagingSource() },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getCustomPostRankingPagingData(): Flowable<PagingData<AmityPost>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true,
                initialLoadSize = (getDefaultPageSize() / 2),
                prefetchDistance = 0
            ),
            queryStreamMediator = CustomPostRankingMediator(),
            pagingSourceFactory = { PostLocalDataStore().getGlobalPostsV5PagingSource() },
            modelMapper = PostModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getPost(postId: String): AmityPost? {
        val post = PostLocalDataStore().getPost(postId)
        return if (post == null) null else PostModelMapper().map(post)
    }

    fun getPosts(postIds: List<String>): List<AmityPost> {
        val posts = if (postIds.isNotEmpty()) {
            PostLocalDataStore().getPosts(postIds).map {
                PostModelMapper().map(it)
            }
        } else {
            emptyList()
        }
        return posts
    }

    fun createPostV4(
        targetType: String,
        targetId: String,
        data: JsonObject,
        attachments: List<AmityPostAttachment>?,
        dataType: AmityPost.DataType? = null,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?,
    ): Single<String> {
        return PostRemoteDataStore().createPost(
            targetType,
            targetId,
            data,
            attachments,
            dataType,
            metadata,
            mentionees
        )
            .flatMap {
                PostQueryPersister().persist(it)
                    .andThen(Single.just(it.posts.first()?.postId ?: ""))
            }
    }

    fun deletePost(postId: String, hardDelete: Boolean): Completable {
        return PostRemoteDataStore().deletePost(postId, hardDelete)
            .flatMapCompletable {
                if (!it.isSuccess) {
                    Completable.complete()
                } else if (hardDelete) {
                    PostLocalDataStore().hardDelete(postId).andThen(Completable.defer {
                        TombstoneRepository().saveTombstone(
                            objectId = postId,
                            tombstoneModelType = TombstoneModelType.POST,
                            amityError = AmityError.ITEM_NOT_FOUND
                        )
                    })
                } else {
                    PostLocalDataStore().softDelete(postId)
                }
            }
    }

    fun approvePost(postId: String): Completable {
        var initialFeedType: AmityFeedType = AmityFeedType.NONE
        return Flowable.fromCallable {
            initialFeedType = PostLocalDataStore().getFeedType(postId)
            PostLocalDataStore().updateFeedType(postId, AmityFeedType.PUBLISHED)
        }.flatMapCompletable {
            PostRemoteDataStore().approvePost(postId).ignoreElement()
        }.doOnError {
            if (AmityError.from(it) == AmityError.FORBIDDEN_ERROR) {
                PostLocalDataStore().updateFeedType(postId, AmityFeedType.NONE)
            } else {
                PostLocalDataStore().updateFeedType(postId, initialFeedType)
            }
        }
    }

    fun declinePost(postId: String): Completable {
        var initialFeedType: AmityFeedType = AmityFeedType.NONE
        return Flowable.fromCallable {
            initialFeedType = PostLocalDataStore().getFeedType(postId)
            PostLocalDataStore().updateFeedType(postId, AmityFeedType.DECLINED)
        }.flatMapCompletable {
            PostRemoteDataStore().declinePost(postId).ignoreElement()
        }.doOnError {
            if (AmityError.from(it) == AmityError.FORBIDDEN_ERROR) {
                PostLocalDataStore().updateFeedType(postId, AmityFeedType.NONE)
            } else {
                PostLocalDataStore().updateFeedType(postId, initialFeedType)
            }
        }
    }

    fun editPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?,
        attachments: List<AmityPostAttachment>?,
    ): Completable {
        return PostRemoteDataStore().editPost(postId, data, metadata, mentionees, attachments)
            .flatMapCompletable {
                val childIds = it.children.map { child -> child.postId }.toMutableList()
                childIds.add(postId) // To avoid empty childIds causing crash on local data store
                PostQueryPersister().persist(it)
                    .andThen(
                        PostLocalDataStore().invalidateChildPosts(postId, childIds)
                    )
            }
    }

    fun flagPost(postId: String, reason: FlagContentRequest): Completable {
        return PostRemoteDataStore().flagPost(postId, reason)
            .flatMapCompletable {
                PostQueryPersister().persist(it)
            }
    }

    fun unFlagPost(postId: String): Completable {
        return PostRemoteDataStore().unFlagPost(postId)
            .flatMapCompletable {
                PostQueryPersister().persist(it)
            }
    }

    fun getLatestPost(
        targetId: String,
        targetType: String,
        includeDeleted: Boolean?,
        postTypes: List<AmityPost.DataType>,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        feedType: AmityFeedType? = null,
    ): Flowable<AmityPost> {
        return PostLocalDataStore().getLatestPosts(
            targetId = targetId,
            targetType = targetType,
            includeDeleted = includeDeleted,
            postTypes = postTypes.map { it.getApiKey() },
            feedType = feedType,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).map {
            PostModelMapper().map(it)
        }
    }

    fun isFlaggedByMe(postId: String): Single<JsonObject> {
        return PostRemoteDataStore().isFlaggedByMe(postId)
    }


}